/**
 * Simple Number Formatter using Decimal.js
 * Indonesian format: dot (.) for thousand separator, comma (,) for decimal
 */

'use strict';

/**
 * Format number as currency with precision
 * @param {number|string} input - Number to format
 * @param {number} maxDecimals - Maximum decimal places (default: 12)
 * @returns {string} Formatted number
 */
function formatCurrency(input, maxDecimals = 12) {
  try {
    // Convert input to Decimal for precision
    let decimal;
    if (typeof input === 'string') {
      // Remove dots (thousand separators) and replace comma (decimal) with dot
      decimal = new Decimal(input.replace(/\./g, '').replace(',', '.'));
    } else {
      decimal = new Decimal(input || 0);
    }

    console.log("decimal", input + " " + maxDecimals);
    // Apply max decimal limit with rounding
    if (decimal.decimalPlaces() > maxDecimals) {
      decimal = decimal.toDecimalPlaces(maxDecimals, Decimal.ROUND_HALF_UP);
      console.log("decimal round", decimal);
    }
    // Get the string representation and remove trailing zeros
    let numStr = decimal.toString();
    // Remove trailing zeros after decimal point
    if (numStr.includes('.')) {
      numStr = numStr.replace(/\.?0+$/, '');
    }

    // Split into integer and decimal parts
    let parts = numStr.split('.');

    // Add dots to integer part (thousand separator)
    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, '.');

    // Join back together with comma as decimal separator
    return parts.join(',');
  } catch (error) {
    console.error('formatCurrency error:', error);
    return '0';
  }
}

/**
 * Add thousand separators to a number for display
 * @param {number|string} value - Number to format
 * @param {number} decimals - Number of decimal places (default: 0, use -1 for auto)
 * @returns {string} Formatted number with thousand separators
 */
window.formatter = window.formatNumber = function (value, decimals = 0) {
  if (value === null || value === undefined || value === '') {
    return '';
  }

  try {
    let decimal = new Decimal(value);

    // Round to specified decimal places if needed
    if (decimals >= 0) {
      decimal = decimal.toDecimalPlaces(decimals, Decimal.ROUND_HALF_UP);
    }

    let numStr = decimal.toString();

    // Remove trailing zeros after decimal point only if decimals is -1 (auto)
    if (decimals === -1 && numStr.includes('.')) {
      numStr = numStr.replace(/\.?0+$/, '');
    }

    // Split into integer and decimal parts
    let parts = numStr.split('.');

    // Add dots to integer part (thousand separator)
    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, '.');

    // Join back together with comma as decimal separator
    return parts.join(',');
  } catch (error) {
    console.error('formatter error:', error);
    return '';
  }
};

/**
 * Parse decimal value from formatted string
 * @param {string|number} value - Formatted value
 * @returns {number} Clean number
 */
function parseDecimalValue(value) {
  if (!value || value === '') return 0;

  // Convert to string if not already
  let str = value.toString().trim();

  // Check the last occurrence of comma or dot to determine which is the decimal separator
  const lastCommaIndex = str.lastIndexOf(',');
  const lastDotIndex = str.lastIndexOf('.');

  // If both exist, the one that comes last is the decimal separator
  if (lastCommaIndex !== -1 && lastDotIndex !== -1) {
    if (lastCommaIndex > lastDotIndex) {
      // Comma is the decimal separator (Indonesian format: 1.000,50)
      // Remove all dots (thousand separators) and replace comma with dot
      str = str.replace(/\./g, '').replace(',', '.');
    } else {
      // Dot is the decimal separator (US format: 1,000.50)
      // Remove all commas (thousand separators)
      str = str.replace(/,/g, '');
    }
  } else if (lastCommaIndex !== -1) {
    // Only comma exists
    // Check if it's a decimal separator (has 1-2 digits after it) or thousand separator
    const afterComma = str.substring(lastCommaIndex + 1);
    if (afterComma.length <= 2 && /^\d+$/.test(afterComma)) {
      // It's a decimal separator (e.g., "10,50")
      str = str.replace(',', '.');
    } else {
      // It's a thousand separator (e.g., "1,000,000")
      str = str.replace(/,/g, '');
    }
  } else if (lastDotIndex !== -1) {
    // Only dot exists - in Indonesian format, dots are ALWAYS thousand separators
    // Remove all dots (e.g., "10.000" → "10000", "2.500.000" → "2500000")
    str = str.replace(/\./g, '');
  }

  return parseFloat(str) || 0;
}

/**
 * Remove thousand separators from formatted number
 * @param {string} formattedValue - Number with thousand separators
 * @returns {number} Clean number without formatting
 */
window.parser = window.parseNumber = function (formattedValue) {
  return parseDecimalValue(formattedValue);
};

/**
 * Format date
 * @param {string|Date} date - Date to format
 * @param {string} format - Moment.js format string
 * @returns {string|null} Formatted date
 */
function formatDate(date, format = "DD/MM/YY") {
  // Try to parse the date using moment
  const momentDate = moment(date);

  // Check if date is valid
  if (!momentDate.isValid()) {
    return null;
  }

  // Return formatted date
  return momentDate.format(format);
}

/**
 * Initialize number formatting on input fields
 */
document.addEventListener('DOMContentLoaded', function () {
  initNumberFormatting();
});

/**
 * Initialize number formatting with jQuery event delegation
 */
window.initNumberFormatting = function () {
  // Use jQuery event delegation for input formatting
  $(document).on('input', 'input[data-format="number"]', function () {
    const maxDecimals = $(this).data('max-decimals') || 0; // Default to 0 for integers

    // Save cursor position and original length
    let cursorPos = this.selectionStart;
    let originalLength = this.value.length;

    // Clean input: remove non-numeric characters except comma (decimal separator)
    let value = this.value.replace(/[^\d,]/g, '');

    // Handle multiple commas (keep only the first one)
    let decimalIndex = value.indexOf(',');
    if (decimalIndex !== -1) {
      value = value.substring(0, decimalIndex + 1) + value.substring(decimalIndex + 1).replace(/,/g, '');
    }

    // Limit decimal places during input
    let parts = value.split(',');
    if (parts[1] && parts[1].length > maxDecimals) {
      parts[1] = parts[1].substring(0, maxDecimals);
      value = parts.join(',');
    }

    // Add dots to integer part (thousand separator)
    if (parts[0]) {
      parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, '.');
      this.value = parts.join(',');
    } else {
      this.value = value;
    }

    // Adjust cursor position
    let newLength = this.value.length;
    if (this.setSelectionRange) {
      this.setSelectionRange(cursorPos + (newLength - originalLength), cursorPos + (newLength - originalLength));
    }
  });
};

// Make functions globally available
window.formatCurrency = formatCurrency;
window.parseDecimalValue = parseDecimalValue;
window.formatDate = formatDate;
