@extends('layouts.layoutMaster')

@section('title', 'Activity Log')

@section('vendor-style')
    <style>
        .activity-timeline {
            position: relative;
            padding-left: 30px;
        }

        .activity-timeline::before {
            content: '';
            position: absolute;
            left: 8px;
            top: 0;
            bottom: 0;
            width: 2px;
            background: #e0e0e0;
        }

        .activity-item {
            position: relative;
            margin-bottom: 25px;
            padding: 20px;
            background: #fff;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            transition: all 0.3s ease;
        }

        .activity-item:hover {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            border-color: #696cff;
        }

        .activity-item::before {
            content: '';
            position: absolute;
            left: -22px;
            top: 25px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #fff;
            border: 3px solid #696cff;
            z-index: 1;
        }

        .activity-item.event-created::before {
            border-color: #28a745;
        }

        .activity-item.event-updated::before {
            border-color: #ffc107;
        }

        .activity-item.event-deleted::before {
            border-color: #dc3545;
        }

        .activity-item.event-approved::before {
            border-color: #28a745;
        }

        .activity-item.event-rejected::before {
            border-color: #dc3545;
        }

        .activity-item.event-canceled::before {
            border-color: #dc3545;
        }

        .activity-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 15px;
        }

        .activity-header-content {
            flex: 1;
        }

        .activity-header-actions {
            flex-shrink: 0;
            margin-left: 15px;
        }

        /* Mobile responsive styles */
        @media (max-width: 768px) {
            .activity-header {
                flex-direction: column;
            }

            .activity-header-actions {
                order: 2;
                margin-left: 0;
                margin-top: 15px;
                width: 100%;
            }

            .activity-header-content {
                order: 1;
            }

            .view-detail-btn {
                width: 100%;
                justify-content: center;
            }

            .activity-timeline {
                padding-left: 20px;
            }

            .activity-timeline::before {
                left: 4px;
            }

            .activity-item::before {
                left: -16px;
                width: 10px;
                height: 10px;
            }

            .activity-meta {
                flex-direction: column;
                gap: 8px;
            }

            .filter-group {
                min-width: 100%;
            }
        }

        .activity-description {
            font-size: 16px;
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
        }

        .activity-meta {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
            font-size: 13px;
            color: #666;
        }

        .activity-meta-item {
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .activity-meta-item i {
            font-size: 14px;
        }

        .activity-changes {
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid #f0f0f0;
        }

        .change-badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 500;
            margin-right: 8px;
            margin-bottom: 5px;
        }

        .change-badge.added {
            background-color: #d4edda;
            color: #155724;
        }

        .change-badge.updated {
            background-color: #fff3cd;
            color: #856404;
        }

        .change-badge.deleted {
            background-color: #f8d7da;
            color: #721c24;
        }

        .filter-card {
            margin-bottom: 25px;
        }

        .filter-row {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
            align-items: end;
        }

        .filter-group {
            flex: 1;
            min-width: 200px;
        }

        .filter-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            font-size: 13px;
            color: #555;
        }

        .filter-group input,
        .filter-group select {
            width: 100%;
        }

        .btn-filter-group {
            display: flex;
            gap: 8px;
        }

        .event-badge {
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .event-badge.created {
            background-color: #d4edda;
            color: #155724;
        }

        .event-badge.updated {
            background-color: #fff3cd;
            color: #856404;
        }

        .event-badge.deleted {
            background-color: #f8d7da;
            color: #721c24;
        }

        .event-badge.approved {
            background-color: #d4edda;
            color: #155724;
        }

        .event-badge.rejected {
            background-color: #f8d7da;
            color: #721c24;
        }

        .event-badge.canceled {
            background-color: #f8d7da;
            color: #721c24;
        }

        .model-badge {
            padding: 4px 10px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 500;
            background-color: #e7e7ff;
            color: #696cff;
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #999;
        }

        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.3;
        }

        .pagination-wrapper {
            margin-top: 30px;
            display: flex;
            justify-content: center;
        }

        .view-detail-btn {
            padding: 6px 15px;
            font-size: 13px;
        }

        /* Modal specific styles */
        #activityDetailModal .modal-body {
            max-height: 75vh;
            overflow-y: auto;
        }

        /* Responsive modal styles */
        @media (max-width: 768px) {
            #activityDetailModal .modal-xl {
                max-width: 95%;
                margin: 0.5rem auto;
            }

            #activityDetailModal .modal-body {
                max-height: 70vh;
                padding: 1rem;
            }

            #activityDetailModal .table-responsive {
                font-size: 0.875rem;
            }

            #activityDetailModal .table th,
            #activityDetailModal .table td {
                padding: 0.5rem;
                word-wrap: break-word;
            }

            #activityDetailModal .row {
                margin-bottom: 1rem;
            }

            #activityDetailModal .col-md-6,
            #activityDetailModal .col-md-12 {
                margin-bottom: 0.75rem;
            }

            #activityDetailModal .card {
                margin-bottom: 0.75rem;
            }

            #activityDetailModal h6 {
                font-size: 0.875rem;
            }

            #activityDetailModal .event-badge,
            #activityDetailModal .change-badge {
                font-size: 0.75rem;
                padding: 3px 8px;
            }
        }

        @media (max-width: 576px) {
            #activityDetailModal .modal-body {
                padding: 0.75rem;
            }

            #activityDetailModal .table {
                font-size: 0.75rem;
            }

            #activityDetailModal .table th,
            #activityDetailModal .table td {
                padding: 0.4rem;
                font-size: 0.75rem;
            }

            /* Stack table columns on very small screens */
            #activityDetailModal .table thead {
                display: none;
            }

            #activityDetailModal .table tr {
                display: block;
                margin-bottom: 1rem;
                border: 1px solid #dee2e6;
                border-radius: 4px;
            }

            #activityDetailModal .table td {
                display: block;
                text-align: left;
                border: none;
                padding: 0.5rem;
            }

            #activityDetailModal .table td:first-child {
                background-color: #f8f9fa;
                font-weight: 600;
                border-bottom: 1px solid #dee2e6;
            }

            #activityDetailModal .table td::before {
                content: attr(data-label);
                font-weight: 600;
                display: block;
                margin-bottom: 0.25rem;
                color: #6c757d;
            }

            #activityDetailModal .table td:first-child::before {
                content: none;
            }
        }
    </style>
@endsection

@php
    $breadcrumbs = [
        ['url' => route('dashboard'), 'label' => 'Dashboard'],
        ['url' => route('activity-log.index'), 'label' => 'Activity Log'],
    ];
@endphp

@section('content')
    <!-- Filter Card -->
    <div class="card filter-card">
        <div class="card-body">
            <h5 class="card-title mb-3">
                <i class="ri-filter-3-line me-2"></i>Filter Activities
            </h5>

            <form method="GET" action="{{ route('activity-log.index') }}" id="filterForm">
                <div class="filter-row d-flex align-items-end">
                    <!-- Search -->
                    <div class="filter-group">
                        <label for="search">Search Description</label>
                        <input type="text" class="form-control" id="search" name="search"
                            value="{{ request('search') }}" placeholder="Search in description...">
                    </div>

                    <!-- Page (Model Type) -->
                    <div class="filter-group">
                        <label for="model_type">Page</label>
                        <select class="form-select" id="model_type" name="model_type">
                            <option value="">All Pages</option>
                            @foreach ($modelTypes as $modelType)
                                <option value="{{ $modelType['value'] }}"
                                    {{ request('model_type') == $modelType['value'] ? 'selected' : '' }}>
                                    {{ $modelType['label'] }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Event Type -->
                    <div class="filter-group">
                        <label for="event">Action Type</label>
                        <select class="form-select" id="event" name="event">
                            <option value="">All Actions</option>
                            @foreach ($events as $event)
                                <option value="{{ $event }}" {{ request('event') == $event ? 'selected' : '' }}>
                                    {{ ucfirst($event) }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <!-- User -->
                    <div class="filter-group">
                        <label for="user_id">User</label>
                        <select class="form-select" id="user_id" name="user_id">
                            <option value="">All Users</option>
                            @foreach ($users as $user)
                                <option value="{{ $user->id }}" {{ request('user_id') == $user->id ? 'selected' : '' }}>
                                    {{ $user->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Start Date -->
                    <div class="filter-group">
                        <label for="start_date">Start Date</label>
                        <input type="date" class="form-control" id="start_date" name="start_date"
                            value="{{ request('start_date') }}">
                    </div>

                    <!-- End Date -->
                    <div class="filter-group">
                        <label for="end_date">End Date</label>
                        <input type="date" class="form-control" id="end_date" name="end_date"
                            value="{{ request('end_date') }}">
                    </div>

                    <!-- Buttons -->
                    <div class="filter-group" style="flex: 0 0 auto;">
                        <div class="btn-filter-group">
                            <button type="submit" class="btn btn-sm btn-primary">
                                Filter
                            </button>
                            <a href="{{ route('activity-log.index') }}" class="btn btn-sm btn-outline-secondary">
                                Reset
                            </a>
                            @can('activity-log.index')
                                <a href="{{ route('activity-log.export', request()->query()) }}" class="btn btn-sm btn-success">
                                    Export
                                </a>
                            @endcan
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Results Summary -->
    <div class="mb-3">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h5 class="mb-0">
                    Activity Timeline
                    <span class="badge bg-label-primary ms-2">{{ $activities->total() }} total</span>
                </h5>
                <small class="text-muted">
                    Showing {{ $activities->firstItem() ?? 0 }} to {{ $activities->lastItem() ?? 0 }} of
                    {{ $activities->total() }} activities
                </small>
            </div>
        </div>
    </div>

    <!-- Activity Timeline -->
    <div class="card">
        <div class="card-body">
            @if ($activities->count() > 0)
                <div class="activity-timeline">
                    @foreach ($activities as $activity)
                        <div class="activity-item event-{{ $activity->event }}">
                            <div class="activity-header">
                                <div class="activity-header-content">
                                    <div class="activity-description">
                                        {{ $activity->description }}
                                    </div>

                                    <div class="activity-meta">
                                        <div class="activity-meta-item">
                                            <i class="ri-calendar-line"></i>
                                            <span>{{ $activity->created_at->format('M d, Y H:i:s') }}</span>
                                        </div>

                                        <div class="activity-meta-item">
                                            <i class="ri-user-line"></i>
                                            <span>{{ $activity->causer ? $activity->causer->name : 'System' }}</span>
                                        </div>

                                        <div class="activity-meta-item">
                                            <span class="event-badge {{ $activity->event }}">{{ $activity->event }}</span>
                                        </div>
                                    </div>
                                </div>

                                <div class="activity-header-actions">
                                    @can('activity-log.show')
                                        <button class="btn btn-sm btn-outline-primary view-detail-btn"
                                            onclick="viewActivityDetail({{ $activity->id }})">
                                            <i class="ri-eye-line me-1"></i>View Details
                                        </button>
                                    @endcan
                                </div>
                            </div>

                            {{-- Quick preview of changes --}}
                            @if ($activity->properties && $activity->properties->count() > 0)
                                <div class="activity-changes">
                                    @php
                                        $summary = \App\Helpers\ActivityLogHelper::getChangeSummary($activity);
                                    @endphp

                                    @if ($summary !== 'No changes recorded')
                                        <small class="text-muted">
                                            <i class="ri-information-line me-1"></i>
                                            {{ $summary }}
                                        </small>
                                    @endif

                                    {{-- Show detail change counts --}}
                                    @if ($activity->properties->has('detail_changes'))
                                        @php
                                            $detailChanges = $activity->properties->get('detail_changes');
                                        @endphp

                                        <div class="mt-2">
                                            @if (!empty($detailChanges['added']))
                                                <span class="change-badge added">
                                                    <i class="ri-add-line"></i>
                                                    {{ count($detailChanges['added']) }} Added
                                                </span>
                                            @endif

                                            @if (!empty($detailChanges['updated']))
                                                <span class="change-badge updated">
                                                    <i class="ri-edit-line"></i>
                                                    {{ count($detailChanges['updated']) }} Updated
                                                </span>
                                            @endif

                                            @if (!empty($detailChanges['deleted']))
                                                <span class="change-badge deleted">
                                                    <i class="ri-delete-bin-line"></i>
                                                    {{ count($detailChanges['deleted']) }} Deleted
                                                </span>
                                            @endif
                                        </div>
                                    @endif
                                </div>
                            @endif
                        </div>
                    @endforeach
                </div>

                <!-- Pagination -->
                <div class="pagination-wrapper">
                    {{ $activities->appends(request()->query())->links() }}
                </div>
            @else
                <div class="empty-state">
                    <i class="ri-history-line"></i>
                    <h4>No Activity Found</h4>
                    <p class="text-muted">
                        @if (request()->hasAny(['search', 'model_type', 'event', 'user_id', 'start_date', 'end_date']))
                            No activities match your current filters. Try adjusting your search criteria.
                        @else
                            No activities have been logged yet.
                        @endif
                    </p>
                </div>
            @endif
        </div>
    </div>

    <!-- Activity Detail Modal -->
    <div class="modal fade" id="activityDetailModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Activity Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="activityDetailContent">
                    <div class="text-center py-5">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('page-script')
    <script>
        function viewActivityDetail(activityId) {
            const modal = new bootstrap.Modal(document.getElementById('activityDetailModal'));
            const content = document.getElementById('activityDetailContent');

            // Show loading
            content.innerHTML = `
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            `;

            modal.show();

            // Fetch activity details
            fetch(`/activity-log/detail/${activityId}`)
                .then(response => response.json())
                .then(data => {
                    content.innerHTML = renderActivityDetail(data);
                })
                .catch(error => {
                    content.innerHTML = `
                        <div class="alert alert-danger">
                            <i class="ri-error-warning-line me-2"></i>
                            Failed to load activity details. Please try again.
                        </div>
                    `;
                });
        }

        function renderActivityDetail(data) {
            let html = `
                <div class="activity-detail-content">
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <h6 class="text-muted mb-2">Description</h6>
                            <p class="mb-0">${data.description}</p>
                        </div>
                        <div class="col-md-6">
                            <h6 class="text-muted mb-2">Action</h6>
                            <span class="event-badge ${data.event}">${data.event}</span>
                        </div>
                    </div>

                    <div class="row mb-4">
                        <div class="col-md-6">
                            <h6 class="text-muted mb-2">Date & Time</h6>
                            <p class="mb-0">${data.created_at}</p>
                        </div>
                        <div class="col-md-6">
                            <h6 class="text-muted mb-2">User</h6>
                            <p class="mb-0">${data.causer}</p>
                        </div>
                    </div>

                    <div class="row mb-4">
                        <div class="col-md-12">
                            <h6 class="text-muted mb-2">Change Summary</h6>
                            <p class="mb-0">${data.change_summary}</p>
                        </div>
                    </div>
            `;

            // Show attribute changes
            if (data.changes && data.changes.attributes) {
                html += `
                    <div class="mb-4">
                        <h6 class="mb-3">Field Changes</h6>
                        <div class="table-responsive">
                            <table class="table table-bordered">
                                <thead>
                                    <tr>
                                        <th style="width: 30%">Field</th>
                                        <th style="width: 35%">Old Value</th>
                                        <th style="width: 35%">New Value</th>
                                    </tr>
                                </thead>
                                <tbody>
                `;

                Object.entries(data.changes.attributes).forEach(([field, change]) => {
                    const fieldLabel = change.label || field;
                    const oldValue = change.old_readable || change.old || 'N/A';
                    const newValue = change.new_readable || change.new || 'N/A';

                    html += `
                        <tr>
                            <td data-label="Field"><strong>${fieldLabel}</strong></td>
                            <td data-label="Old Value">${oldValue}</td>
                            <td data-label="New Value">${newValue}</td>
                        </tr>
                    `;
                });

                html += `
                                </tbody>
                            </table>
                        </div>
                    </div>
                `;
            }

            // Show added details
            if (data.changes && data.changes.details_added) {
                html += `
                    <div class="mb-4">
                        <h6 class="mb-3">
                            <span class="change-badge added">
                                <i class="ri-add-line"></i> ${data.changes.details_added.length} Details Added
                            </span>
                        </h6>
                `;

                data.changes.details_added.forEach((detail, index) => {
                    html += `
                        <div class="card mb-3">
                            <div class="card-header">
                                <strong>Detail #${index + 1}</strong>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-bordered mb-0">
                                        <thead>
                                            <tr>
                                                <th style="width: 30%">Field</th>
                                                <th style="width: 70%">Value</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                    `;

                    detail.fields.forEach(field => {
                        html += `
                            <tr>
                                <td data-label="Field"><strong>${field.label}</strong></td>
                                <td data-label="Value">${field.value_readable}</td>
                            </tr>
                        `;
                    });

                    html += `
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    `;
                });

                html += `</div>`;
            }

            // Show updated details
            if (data.changes && data.changes.details_updated) {
                html += `
                    <div class="mb-4">
                        <h6 class="mb-3">
                            <span class="change-badge updated">
                                <i class="ri-edit-line"></i> ${data.changes.details_updated.length} Details Updated
                            </span>
                        </h6>
                `;

                data.changes.details_updated.forEach((detail, index) => {
                    html += `
                        <div class="card mb-3">
                            <div class="card-header">
                                <strong>Detail #${index + 1}</strong>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-bordered mb-0">
                                        <thead>
                                            <tr>
                                                <th style="width: 30%">Field</th>
                                                <th style="width: 35%">Old Value</th>
                                                <th style="width: 35%">New Value</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                    `;

                    detail.fields.forEach(field => {
                        html += `
                            <tr>
                                <td data-label="Field"><strong>${field.label}</strong></td>
                                <td data-label="Old Value">${field.old_readable}</td>
                                <td data-label="New Value">${field.new_readable}</td>
                            </tr>
                        `;
                    });

                    html += `
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    `;
                });

                html += `</div>`;
            }

            // Show deleted details
            if (data.changes && data.changes.details_deleted) {
                html += `
                    <div class="mb-4">
                        <h6 class="mb-3">
                            <span class="change-badge deleted">
                                <i class="ri-delete-bin-line"></i> ${data.changes.details_deleted.length} Details Deleted
                            </span>
                        </h6>
                `;

                data.changes.details_deleted.forEach((detail, index) => {
                    html += `
                        <div class="card mb-3">
                            <div class="card-header">
                                <strong>Detail #${index + 1}</strong>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-bordered mb-0">
                                        <thead>
                                            <tr>
                                                <th style="width: 30%">Field</th>
                                                <th style="width: 70%">Value</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                    `;

                    detail.fields.forEach(field => {
                        html += `
                            <tr>
                                <td data-label="Field"><strong>${field.label}</strong></td>
                                <td data-label="Value">${field.value_readable}</td>
                            </tr>
                        `;
                    });

                    html += `
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    `;
                });

                html += `</div>`;
            }

            // Show added permissions
            if (data.changes && data.changes.permissions_added) {
                // Use grouped count if available, otherwise count the array length
                const addedCount = data.changes.permissions_added_count || data.changes.permissions_added.length;

                html += `
                    <div class="mb-4">
                        <h6 class="mb-3">
                            <span class="change-badge added">
                                <i class="ri-shield-check-line"></i> ${addedCount} Permission(s) Added
                            </span>
                        </h6>
                        <div class="card">
                            <div class="card-body">
                                <ul class="mb-0">
                `;

                data.changes.permissions_added.forEach(permission => {
                    html += `<li class="text-success"><strong>${permission}</strong></li>`;
                });

                html += `
                                </ul>
                            </div>
                        </div>
                    </div>
                `;
            }

            // Show removed permissions
            if (data.changes && data.changes.permissions_removed) {
                // Use grouped count if available, otherwise count the array length
                const removedCount = data.changes.permissions_removed_count || data.changes.permissions_removed.length;

                html += `
                    <div class="mb-4">
                        <h6 class="mb-3">
                            <span class="change-badge deleted">
                                <i class="ri-shield-cross-line"></i> ${removedCount} Permission(s) Removed
                            </span>
                        </h6>
                        <div class="card">
                            <div class="card-body">
                                <ul class="mb-0">
                `;

                data.changes.permissions_removed.forEach(permission => {
                    html += `<li class="text-danger">${permission}</li>`;
                });

                html += `
                                </ul>
                            </div>
                        </div>
                    </div>
                `;
            }

            html += `</div>`;

            return html;
        }
    </script>
@endsection
