@extends('layouts.layoutMaster')

@section('title', ($model->id ? 'Update' : 'Create') . ' Journal')

@section('vendor-style')
    @vite('resources/assets/vendor/libs/select2/select2.scss')
@endsection

@section('vendor-script')
    @vite('resources/assets/vendor/libs/select2/select2.js')
@endsection

@section('page-script')
    @vite('resources/assets/js/forms-selects.js')
@endsection

@php
$m = 'App\\Models\\Journal';
$labels = Helper::getModelLabels($m);

$breadcrumbs = [
        ['url' => route('dashboard'), 'label' => 'Dashboard'],
        ['url' => route('journal.index'), 'label' => 'Journal'],
    ];

    if ($model->id) {
        $breadcrumbs[] = [
            'url' => route('journal.show', ['journal' => $model->id]),
            'label' => $model->code,
        ];
        $breadcrumbs[] = [
            'url' => route('journal.edit', ['journal' => $model->id]),
            'label' => 'Update Journal',
        ];
    } else {
        $breadcrumbs[] = [
            'url' => route('journal.create'),
            'label' => 'Create Journal',
        ];
    }

@endphp

@section('content')
    <div class="row form-row-with-sticky-actions">
        <div class="col-12">
            <div class="card shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h5 class="mb-1">{{ $model->id ? 'Edit' : 'New' }} Journal</h5>
                            <p class="text-muted mb-0 small">
                                {{ $model->id ? 'Update journal information' : 'Fill in the details below to create a new journal' }}
                            </p>
                        </div>
                    </div>
                </div>
                <div class="card-body pt-4">
                    <form id="journal-form" method="POST"
                        action="{{ $model->id ? route('journal.update', ['journal' => $model->id]) : route('journal.store') }}">
                        @csrf
                        @if ($model->id)
                            @method('PUT')
                        @endif

                        <div class="row g-3">
                            <!-- Date -->
                            <div class="col-md-2">
                                <label for="date" class="form-label">
                                    {{ $labels['date'] ?? 'Date' }} <span class="text-danger">*</span>
                                </label>
                                <input type="date" class="form-control @error('date') is-invalid @enderror"
                                    id="date" name="date" value="{{ old('date', $model->date?->format('Y-m-d')) }}"
                                    required autofocus>
                                @error('date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Description -->
                            <div class="col-md-10">
                                <label for="description" class="form-label">{{ $labels['description'] ?? 'Description' }}</label>
                                <input type="text" class="form-control @error('description') is-invalid @enderror"
                                    id="description" name="description"
                                    value="{{ old('description', $model->description) }}" placeholder="Enter description">
                                @error('description')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Divider -->
                            <div class="col-12">
                                <div class="border-top pt-3 mt-2">
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <h6 class="text-muted mb-0">Account Journal Details</h6>
                                        <button type="button" class="btn btn-sm btn-primary" id="add-detail-btn">
                                            <i class="icon-base ti tabler-plus"></i> Add Detail
                                        </button>
                                    </div>
                                </div>
                            </div>

                            <!-- Details Table -->
                            <div class="col-12">
                                @error('details')
                                    <div class="alert alert-danger">{{ $message }}</div>
                                @enderror
                                <div class="table-responsive">
                                    <table class="table table-bordered" id="details-table">
                                        <thead class="table-light">
                                            <tr>
                                                <th style="width: 28%;">Account <span class="text-danger">*</span></th>
                                                <th style="width: 18%;">Debit</th>
                                                <th style="width: 18%;">Credit</th>
                                                <th style="width: 29%;">Description</th>
                                                <th style="width: 7%;">Action</th>
                                            </tr>
                                        </thead>
                                        <tbody id="details-tbody">
                                            @php
                                                $details = old('details', $model->accountJournals ?? []);
                                                if (empty($details)) {
                                                    $details = [];
                                                }
                                            @endphp
                                            @foreach ($details as $index => $detail)
                                                <tr>
                                                    @if(is_object($detail) && isset($detail->id))
                                                        <input type="hidden" name="details[{{ $index }}][id]" value="{{ $detail->id }}">
                                                    @endif
                                                    <td>
                                                        <select name="details[{{ $index }}][id_accounts]"
                                                            class="form-select select2 @error('details.' . $index . '.id_accounts') is-invalid @enderror"
                                                            required>
                                                            <option value="">Select Account</option>
                                                            @foreach ($accounts as $account)
                                                                <option value="{{ $account->id }}"
                                                                    {{ old('details.' . $index . '.id_accounts', $detail->id_accounts ?? '') == $account->id ? 'selected' : '' }}>
                                                                    {{ $account->code }} - {{ $account->name }}
                                                                </option>
                                                            @endforeach
                                                        </select>
                                                        @error('details.' . $index . '.id_accounts')
                                                            <div class="invalid-feedback">{{ $message }}</div>
                                                        @enderror
                                                    </td>
                                                    <td>
                                                        <input type="text"
                                                            name="details[{{ $index }}][debit]"
                                                            class="form-control @error('details.' . $index . '.debit') is-invalid @enderror debit-input"
                                                            value="{{ old('details.' . $index . '.debit', Helper::formatNumber($detail->debit ?? 0)) }}"
                                                            placeholder="0" data-format="number" data-submit-raw="true">
                                                        @error('details.' . $index . '.debit')
                                                            <div class="invalid-feedback">{{ $message }}</div>
                                                        @enderror
                                                    </td>
                                                    <td>
                                                        <input type="text"
                                                            name="details[{{ $index }}][credit]"
                                                            class="form-control @error('details.' . $index . '.credit') is-invalid @enderror credit-input"
                                                            value="{{ old('details.' . $index . '.credit', Helper::formatNumber($detail->credit ?? 0)) }}"
                                                            placeholder="0" data-format="number" data-submit-raw="true">
                                                        @error('details.' . $index . '.credit')
                                                            <div class="invalid-feedback">{{ $message }}</div>
                                                        @enderror
                                                    </td>
                                                    <td>
                                                        <input type="text"
                                                            name="details[{{ $index }}][description]"
                                                            class="form-control @error('details.' . $index . '.description') is-invalid @enderror"
                                                            value="{{ old('details.' . $index . '.description', $detail->description ?? '') }}"
                                                            placeholder="Description">
                                                        @error('details.' . $index . '.description')
                                                            <div class="invalid-feedback">{{ $message }}</div>
                                                        @enderror
                                                    </td>
                                                    <td class="text-center">
                                                        <button type="button"
                                                            class="btn btn-sm btn-danger remove-detail-btn">
                                                            <i class="icon-base ti tabler-trash"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                        <tfoot>
                                            <tr>
                                                <td class="text-end"><strong>Total:</strong></td>
                                                <td><strong id="total-debit">0.00</strong></td>
                                                <td><strong id="total-credit">0.00</strong></td>
                                                <td colspan="2">
                                                    <span id="balance-warning" class="text-danger small" style="display: none;">
                                                        <i class="icon-base ti tabler-alert-circle"></i> Debit and Credit are not balanced!
                                                    </span>
                                                    <span id="balance-ok" class="text-success small" style="display: none;">
                                                        <i class="icon-base ti tabler-check"></i> Balanced
                                                    </span>
                                                </td>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Form Actions - Sticky to bottom -->
        <div class="form-actions-sticky">
            <a href="{{ route('journal.index') }}" class="btn btn-sm btn-label-secondary">
                Cancel
            </a>
            <button type="submit" form="journal-form" class="btn btn-sm btn-primary">
                {{ $model->id ? 'Update' : 'Create' }} Journal
            </button>
        </div>
    @endsection

    @push('scripts')
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                let detailIndex = {{ count($details) }};

                const accountsOptions = `
            <option value="">Select Account</option>
            @foreach ($accounts as $account)
                <option value="{{ $account->id }}">{{ $account->code }} - {{ $account->name }}</option>
            @endforeach
        `;

                function parseFormattedNumber(value) {
                    if (!value) return 0;
                    // Remove thousand separators (dots) and replace decimal comma with dot
                    return parseFloat(value.toString().replace(/\./g, '').replace(',', '.')) || 0;
                }

                function calculateTotals() {
                    let totalDebit = 0;
                    let totalCredit = 0;

                    document.querySelectorAll('.debit-input').forEach(function(input) {
                        const value = parseFormattedNumber(input.value);
                        totalDebit += value;
                    });

                    document.querySelectorAll('.credit-input').forEach(function(input) {
                        const value = parseFormattedNumber(input.value);
                        totalCredit += value;
                    });

                    document.getElementById('total-debit').textContent = totalDebit.toLocaleString('id-ID', {
                        minimumFractionDigits: 2,
                        maximumFractionDigits: 2
                    });
                    document.getElementById('total-credit').textContent = totalCredit.toLocaleString('id-ID', {
                        minimumFractionDigits: 2,
                        maximumFractionDigits: 2
                    });

                    // Check balance
                    const balanceWarning = document.getElementById('balance-warning');
                    const balanceOk = document.getElementById('balance-ok');
                    const difference = Math.abs(totalDebit - totalCredit);

                    if (difference > 0.01) {
                        balanceWarning.style.display = 'inline';
                        balanceOk.style.display = 'none';
                    } else if (totalDebit > 0 || totalCredit > 0) {
                        balanceWarning.style.display = 'none';
                        balanceOk.style.display = 'inline';
                    } else {
                        balanceWarning.style.display = 'none';
                        balanceOk.style.display = 'none';
                    }
                }

                function addDetailRow() {
                    const tbody = document.getElementById('details-tbody');
                    const newRow = document.createElement('tr');

                    newRow.innerHTML = `
                <td>
                    <select name="details[${detailIndex}][id_accounts]" class="form-select select2" required>
                        ${accountsOptions}
                    </select>
                </td>
                <td>
                    <input type="text" name="details[${detailIndex}][debit]" class="form-control debit-input" value="0" placeholder="0" data-format="number" data-submit-raw="true">
                </td>
                <td>
                    <input type="text" name="details[${detailIndex}][credit]" class="form-control credit-input" value="0" placeholder="0" data-format="number" data-submit-raw="true">
                </td>
                <td>
                    <input type="text" name="details[${detailIndex}][description]" class="form-control" placeholder="Description">
                </td>
                <td class="text-center">
                    <button type="button" class="btn btn-sm btn-danger remove-detail-btn">
                        <i class="icon-base ti tabler-trash"></i>
                    </button>
                </td>
            `;

                    tbody.appendChild(newRow);
                    detailIndex++;
                    calculateTotals();
                }

                // Add detail button
                document.getElementById('add-detail-btn').addEventListener('click', function() {
                    addDetailRow();
                });

                // Remove detail button (event delegation)
                document.addEventListener('click', function(e) {
                    if (e.target.closest('.remove-detail-btn')) {
                        const row = e.target.closest('tr');
                        const tbody = document.getElementById('details-tbody');

                        if (tbody.querySelectorAll('tr').length > 1) {
                            row.remove();
                            calculateTotals();
                        } else {
                            alert('At least one detail row is required');
                        }
                    }
                });

                // Calculate totals on input change
                document.addEventListener('input', function(e) {
                    if (e.target.classList.contains('debit-input') || e.target.classList.contains(
                            'credit-input')) {
                        calculateTotals();
                    }
                });

                // Form submission validation
                document.getElementById('journal-form').addEventListener('submit', function(e) {
                    let totalDebit = 0;
                    let totalCredit = 0;

                    document.querySelectorAll('.debit-input').forEach(function(input) {
                        totalDebit += parseFormattedNumber(input.value);
                    });

                    document.querySelectorAll('.credit-input').forEach(function(input) {
                        totalCredit += parseFormattedNumber(input.value);
                    });

                    const difference = Math.abs(totalDebit - totalCredit);

                    if (difference > 0.01) {
                        e.preventDefault();
                        alert('Total debit and credit must be balanced!\n\nDebit: ' + totalDebit.toFixed(2) + '\nCredit: ' + totalCredit.toFixed(2) + '\nDifference: ' + difference.toFixed(2));
                        return false;
                    }

                    const tbody = document.getElementById('details-tbody');
                    if (tbody.querySelectorAll('tr').length === 0) {
                        e.preventDefault();
                        alert('At least one account journal detail is required.');
                        return false;
                    }
                });

                // Initial calculation
                calculateTotals();
            });
        </script>
    @endpush
