@extends('layouts.layoutMaster')

@section('title', ($model->id ? 'Update' : 'Create') . ' Role')

@php
use Illuminate\Support\Str;

$m = 'App\\Models\\Role';
$labels = Helper::getModelLabels($m);

$breadcrumbs = [
        ['url' => route('dashboard'), 'label' => 'Dashboard'],
        ['url' => route('role.index'), 'label' => 'Role'],
    ];

    if ($model->id) {
        $breadcrumbs[] = [
            'url' => route('role.show', ['role' => $model->id]),
            'label' => $model->name,
        ];
        $breadcrumbs[] = [
            'url' => route('role.edit', ['role' => $model->id]),
            'label' => 'Update Role',
        ];
    } else {
        $breadcrumbs[] = [
            'url' => route('role.create'),
            'label' => 'Create Role',
        ];
    }

    // Get existing permissions for the role
    $existingPermissions = $model->id ? $model->permissions->pluck('name')->toArray() : [];
@endphp

@section('content')
    <div class="row form-row-with-sticky-actions">
        <div class="col-12">
            <div class="card">
                <div class="card-header border-bottom">
                    <h5 class="mb-0">{{ $model->id ? 'Edit' : 'New' }} Role</h5>
                </div>
                <div class="card-body">
                    <form id="role-form" method="POST"
                        action="{{ $model->id ? route('role.update', ['role' => $model->id]) : route('role.store') }}">
                        @csrf
                        @if ($model->id)
                            @method('PUT')
                        @endif

                        <!-- Role Name -->
                        <div class="mb-4">
                            <label for="name" class="form-label fw-semibold">
                                {{ $labels['name'] ?? 'Role Name' }} <span class="text-danger">*</span>
                            </label>
                            <input type="text" class="form-control @error('name') is-invalid @enderror" id="name"
                                name="name" value="{{ old('name', $model->name) }}" required autofocus
                                placeholder="e.g., Admin, Manager, Staff">
                            @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Permissions Section -->
                        <div class="mt-4">
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <label class="form-label fw-semibold mb-0">Permissions</label>
                                <div>
                                    <button type="button" class="btn btn-sm btn-outline-primary me-2" id="select-all">
                                        <i class="bx bx-check-double"></i> Select All
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-secondary" id="deselect-all">
                                        <i class="bx bx-x"></i> Clear All
                                    </button>
                                </div>
                            </div>

                            <!-- Basic Permissions -->
                            <h6 class="fw-semibold text-primary mb-3">Basic Permission</h6>
                            <div class="table-responsive mb-4">
                                <table class="table table-bordered align-middle">
                                    <thead class="table-light">
                                        <tr>
                                            <th style="width: 25%;" class="fw-semibold">Module</th>
                                            <th style="width: 15%;" class="text-center fw-semibold">
                                                <div class="form-check d-inline-block mb-0">
                                                    <input class="form-check-input" type="checkbox" id="select-all-index">
                                                    <label class="form-check-label" for="select-all-index">View All</label>
                                                </div>
                                            </th>
                                            <th style="width: 15%;" class="text-center fw-semibold">
                                                <div class="form-check d-inline-block mb-0">
                                                    <input class="form-check-input" type="checkbox" id="select-all-show">
                                                    <label class="form-check-label" for="select-all-show">View
                                                        Detail</label>
                                                </div>
                                            </th>
                                            <th style="width: 15%;" class="text-center fw-semibold">
                                                <div class="form-check d-inline-block mb-0">
                                                    <input class="form-check-input" type="checkbox" id="select-all-create">
                                                    <label class="form-check-label" for="select-all-create">Create</label>
                                                </div>
                                            </th>
                                            <th style="width: 15%;" class="text-center fw-semibold">
                                                <div class="form-check d-inline-block mb-0">
                                                    <input class="form-check-input" type="checkbox" id="select-all-update">
                                                    <label class="form-check-label" for="select-all-update">Update</label>
                                                </div>
                                            </th>
                                            <th style="width: 15%;" class="text-center fw-semibold">
                                                <div class="form-check d-inline-block mb-0">
                                                    <input class="form-check-input" type="checkbox" id="select-all-destroy">
                                                    <label class="form-check-label" for="select-all-destroy">Delete</label>
                                                </div>
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @foreach ($permissions as $resourceLabel => $resourcePermissions)
                                            <tr>
                                                <td class="fw-medium">
                                                    <div class="form-check">
                                                        <input class="form-check-input select-module-basic" type="checkbox"
                                                            data-module="{{ Str::slug($resourceLabel) }}"
                                                            id="module-basic-{{ Str::slug($resourceLabel) }}">
                                                        <label class="form-check-label fw-semibold"
                                                            for="module-basic-{{ Str::slug($resourceLabel) }}">
                                                            {{ $resourceLabel }}
                                                        </label>
                                                    </div>
                                                </td>
                                                @php
                                                    // Create a map of permissions by key for easy lookup
                                                    $permissionMap = collect($resourcePermissions)->keyBy('key');
                                                    $standardActions = ['index', 'show', 'create', 'update', 'destroy'];
                                                @endphp
                                                @foreach ($standardActions as $actionKey)
                                                    @if ($permissionMap->has($actionKey))
                                                        @php
                                                            $permission = $permissionMap[$actionKey];
                                                            $isChecked = false;
                                                            foreach ($permission['actual_permissions'] as $actualPerm) {
                                                                if (in_array($actualPerm, $existingPermissions)) {
                                                                    $isChecked = true;
                                                                    break;
                                                                }
                                                            }
                                                            $checkboxId =
                                                                'permission-' .
                                                                str_replace(
                                                                    ['.', '-'],
                                                                    '_',
                                                                    $permission['resource_key'] . '-' . $permission['key'],
                                                                );
                                                        @endphp
                                                        <td class="text-center">
                                                            <div class="form-check d-inline-block mb-0">
                                                                <input
                                                                    class="form-check-input permission-checkbox basic-permission permission-type-{{ $permission['key'] }} module-{{ Str::slug($resourceLabel) }}"
                                                                    type="checkbox"
                                                                    name="permissions[{{ $permission['resource_key'] }}_{{ $permission['key'] }}][]"
                                                                    value="{{ json_encode($permission['actual_permissions']) }}"
                                                                    id="{{ $checkboxId }}"
                                                                    data-permissions="{{ json_encode($permission['actual_permissions']) }}"
                                                                    data-module="{{ Str::slug($resourceLabel) }}"
                                                                    {{ $isChecked ? 'checked' : '' }}>
                                                                <label class="form-check-label visually-hidden"
                                                                    for="{{ $checkboxId }}">
                                                                    {{ $permission['label'] }}
                                                                </label>
                                                            </div>
                                                        </td>
                                                    @else
                                                        <td class="text-center">
                                                            <span class="text-muted">—</span>
                                                        </td>
                                                    @endif
                                                @endforeach
                                            </tr>
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>

                            <!-- Custom Permissions -->
                            <h6 class="fw-semibold text-primary mb-3">Custom Permission</h6>
                            <div class="table-responsive">
                                <table class="table table-bordered align-middle">
                                    <thead class="table-light">
                                        <tr>
                                            <th style="width: 25%;" class="fw-semibold">Module</th>
                                            <th class="fw-semibold">Custom Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @foreach ($permissions as $resourceLabel => $resourcePermissions)
                                            @php
                                                $permission = $resourcePermissions[0] ?? null;
                                                $resourceKey = $permission['resource_key'] ?? Str::slug($resourceLabel);
                                                $existingCustom = $customPermissions[$resourceKey] ?? [];
                                            @endphp
                                            @if (!empty($existingCustom))
                                                <tr>
                                                    <td class="fw-medium">
                                                        <div class="form-check">
                                                            <input class="form-check-input select-module-custom"
                                                                type="checkbox"
                                                                data-module="{{ Str::slug($resourceLabel) }}"
                                                                id="module-custom-{{ Str::slug($resourceLabel) }}">
                                                            <label class="form-check-label fw-semibold"
                                                                for="module-custom-{{ Str::slug($resourceLabel) }}">
                                                                {{ $resourceLabel }}
                                                            </label>
                                                        </div>
                                                    </td>
                                                    <td>
                                                        <div class="custom-actions-container"
                                                            data-module="{{ Str::slug($resourceLabel) }}">
                                                            @foreach ($existingCustom as $customPerm)
                                                                @php
                                                                    $customCheckboxId =
                                                                        'custom-' .
                                                                        str_replace('.', '-', $customPerm['name']);
                                                                    $isCustomChecked = in_array(
                                                                        $customPerm['name'],
                                                                        $existingPermissions,
                                                                    );
                                                                @endphp
                                                                <div class="form-check form-check-inline mb-1">
                                                                    <input
                                                                        class="form-check-input permission-checkbox custom-permission module-{{ Str::slug($resourceLabel) }}"
                                                                        type="checkbox"
                                                                        name="permissions[{{ $resourceKey }}_custom_{{ $customPerm['action'] }}][]"
                                                                        value='["{{ $customPerm['name'] }}"]'
                                                                        id="{{ $customCheckboxId }}"
                                                                        data-permissions='["{{ $customPerm['name'] }}"]'
                                                                        data-module="{{ Str::slug($resourceLabel) }}"
                                                                        {{ $isCustomChecked ? 'checked' : '' }}>
                                                                    <label class="form-check-label"
                                                                        for="{{ $customCheckboxId }}">
                                                                        {{ $customPerm['label'] }}
                                                                    </label>
                                                                </div>
                                                            @endforeach
                                                        </div>
                                                    </td>
                                                </tr>
                                            @endif
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>
                            @error('permissions')
                                <div class="text-danger small mt-2">{{ $message }}</div>
                            @enderror
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Form Actions - Sticky to bottom -->
    <div class="form-actions-sticky">
        <a href="{{ route('role.index') }}" class="btn btn-sm btn-label-secondary">
            Cancel
        </a>
        <button type="submit" form="role-form" class="btn btn-sm btn-primary">
            {{ $model->id ? 'Update' : 'Create' }} Role
        </button>
    </div>
@endsection

@section('page-script')
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Role form JS loaded');

            // Parse JSON data from checkboxes and set values properly
            document.querySelectorAll('.permission-checkbox').forEach(function(checkbox) {
                const permissions = checkbox.dataset.permissions;
                if (permissions) {
                    checkbox.value = permissions;
                }
            });

            // Function to update module checkbox state for basic permissions
            function updateModuleBasicState(module) {
                const moduleCheckbox = document.querySelector(`.select-module-basic[data-module="${module}"]`);
                const permissionCheckboxes = document.querySelectorAll(`.basic-permission.module-${module}`);

                if (!moduleCheckbox || permissionCheckboxes.length === 0) return;

                const total = permissionCheckboxes.length;
                const checked = Array.from(permissionCheckboxes).filter(cb => cb.checked).length;

                moduleCheckbox.checked = total === checked;
            }

            // Function to update module checkbox state for custom permissions
            function updateModuleCustomState(module) {
                const moduleCheckbox = document.querySelector(`.select-module-custom[data-module="${module}"]`);
                const permissionCheckboxes = document.querySelectorAll(`.custom-permission.module-${module}`);

                if (!moduleCheckbox || permissionCheckboxes.length === 0) return;

                const total = permissionCheckboxes.length;
                const checked = Array.from(permissionCheckboxes).filter(cb => cb.checked).length;

                moduleCheckbox.checked = total === checked;
            }

            // Function to update column header checkbox state
            function updateColumnState(permissionType) {
                const columnCheckbox = document.getElementById(`select-all-${permissionType}`);
                const typeCheckboxes = document.querySelectorAll(`.permission-type-${permissionType}`);

                if (!columnCheckbox || typeCheckboxes.length === 0) return;

                const total = typeCheckboxes.length;
                const checked = Array.from(typeCheckboxes).filter(cb => cb.checked).length;

                console.log(`Updating column ${permissionType}: ${checked}/${total} checked`);

                columnCheckbox.checked = total === checked;
            }

            // Select All button
            const selectAllBtn = document.getElementById('select-all');
            if (selectAllBtn) {
                selectAllBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    console.log('Select all clicked');
                    document.querySelectorAll('.permission-checkbox').forEach(cb => cb.checked = true);
                    document.querySelectorAll('.select-module-basic').forEach(cb => cb.checked = true);
                    document.querySelectorAll('.select-module-custom').forEach(cb => cb.checked = true);
                    document.querySelectorAll('[id^="select-all-"]').forEach(cb => cb.checked = true);
                });
            }

            // Deselect All button
            const deselectAllBtn = document.getElementById('deselect-all');
            if (deselectAllBtn) {
                deselectAllBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    console.log('Deselect all clicked');
                    document.querySelectorAll('.permission-checkbox').forEach(cb => cb.checked = false);
                    document.querySelectorAll('.select-module-basic').forEach(cb => {
                        cb.checked = false;
                        cb.indeterminate = false;
                    });
                    document.querySelectorAll('.select-module-custom').forEach(cb => {
                        cb.checked = false;
                        cb.indeterminate = false;
                    });
                    document.querySelectorAll('[id^="select-all-"]').forEach(cb => {
                        cb.checked = false;
                        cb.indeterminate = false;
                    });
                });
            }

            let isUpdating = false; // Flag to prevent circular updates

            // Select all by column (permission type) - only affects basic permissions
            document.querySelectorAll('[id^="select-all-"]').forEach(function(columnCheckbox) {
                columnCheckbox.addEventListener('change', function() {
                    if (isUpdating) return;

                    const permissionType = this.id.replace('select-all-', '');
                    const isChecked = this.checked;

                    console.log(`Column ${permissionType} changed to ${isChecked}`);

                    isUpdating = true;

                    // Check/uncheck all permission checkboxes of this type
                    document.querySelectorAll(`.permission-type-${permissionType}`).forEach(cb => {
                        cb.checked = isChecked;
                    });

                    // Update module row checkboxes based on their permissions
                    const uniqueModules = new Set();
                    document.querySelectorAll(`.permission-type-${permissionType}`).forEach(cb => {
                        const module = cb.dataset.module;
                        if (module) uniqueModules.add(module);
                    });

                    uniqueModules.forEach(module => updateModuleBasicState(module));

                    isUpdating = false;
                });
            });

            // Select all basic permissions for a module (row checkbox)
            document.querySelectorAll('.select-module-basic').forEach(function(moduleCheckbox) {
                moduleCheckbox.addEventListener('change', function() {
                    if (isUpdating) return;

                    const module = this.dataset.module;
                    const isChecked = this.checked;

                    console.log(`Module basic ${module} changed to ${isChecked}`);

                    isUpdating = true;

                    // Check/uncheck all basic permission checkboxes for this module
                    document.querySelectorAll(`.module-${module}.basic-permission`).forEach(cb => {
                        cb.checked = isChecked;
                    });

                    isUpdating = false;

                    // Manually trigger column updates after checkboxes are updated
                    ['index', 'show', 'create', 'update', 'destroy'].forEach(type => {
                        updateColumnState(type);
                    });
                });
            });

            // Select all custom permissions for a module (row checkbox)
            document.querySelectorAll('.select-module-custom').forEach(function(moduleCheckbox) {
                moduleCheckbox.addEventListener('change', function() {
                    if (isUpdating) return;

                    const module = this.dataset.module;
                    const isChecked = this.checked;

                    console.log(`Module custom ${module} changed to ${isChecked}`);

                    isUpdating = true;

                    // Check/uncheck all custom permission checkboxes for this module
                    document.querySelectorAll(`.module-${module}.custom-permission`).forEach(cb => {
                        cb.checked = isChecked;
                    });

                    isUpdating = false;
                });
            });

            // Update module and column checkboxes when individual permissions change
            document.querySelectorAll('.permission-checkbox').forEach(function(checkbox) {
                checkbox.addEventListener('change', function() {
                    if (isUpdating) return;

                    const module = this.dataset.module;
                    const classes = this.className;
                    const typeMatch = classes.match(/permission-type-(\w+)/);
                    const isBasic = classes.includes('basic-permission');
                    const isCustom = classes.includes('custom-permission');

                    isUpdating = true;

                    if (module) {
                        if (isBasic) {
                            updateModuleBasicState(module);
                        }
                        if (isCustom) {
                            updateModuleCustomState(module);
                        }
                    }

                    if (typeMatch && typeMatch[1]) {
                        updateColumnState(typeMatch[1]);
                    }

                    isUpdating = false;
                });
            });

            // Handle form submission to properly format permissions
            const roleForm = document.getElementById('role-form');
            if (roleForm) {
                roleForm.addEventListener('submit', function(e) {
                    console.log('Form submitting...');

                    document.querySelectorAll('.permission-checkbox:checked').forEach(function(checkbox) {
                        try {
                            const perms = JSON.parse(checkbox.value);
                            checkbox.removeAttribute('data-permissions');

                            const baseName = checkbox.name.replace('[]', '');
                            perms.forEach(function(perm, index) {
                                const hiddenInput = document.createElement('input');
                                hiddenInput.type = 'hidden';
                                hiddenInput.name = baseName + `[${index}]`;
                                hiddenInput.value = perm;
                                roleForm.appendChild(hiddenInput);
                            });

                            checkbox.disabled = true;
                        } catch (error) {
                            console.error('Error processing permission:', error, checkbox);
                        }
                    });
                });
            }

            // Initialize all checkbox states on page load
            document.querySelectorAll('.select-module-basic').forEach(function(moduleCheckbox) {
                const module = moduleCheckbox.dataset.module;
                updateModuleBasicState(module);
            });

            document.querySelectorAll('.select-module-custom').forEach(function(moduleCheckbox) {
                const module = moduleCheckbox.dataset.module;
                updateModuleCustomState(module);
            });

            ['index', 'show', 'create', 'update', 'destroy'].forEach(type => {
                updateColumnState(type);
            });

            console.log('Role form JS initialized');
        });
    </script>

    <style>
        .table-bordered th,
        .table-bordered td {
            border: 1px solid #d9dee3;
            vertical-align: middle;
        }

        .table thead th {
            background-color: #f8f9fa;
            font-size: 0.8125rem;
        }

        .form-check-label {
            cursor: pointer;
            user-select: none;
        }

        .custom-actions-container {
            min-height: 32px;
        }

        .table tbody tr:hover {
            background-color: rgba(67, 89, 113, 0.02);
        }

        .form-check-inline {
            margin-right: 0.5rem;
        }
    </style>
@endsection
