@extends('layouts.layoutMaster')

@section('title', 'Role Detail')

@php
    $breadcrumbs = [
        ['url' => route('dashboard'), 'label' => 'Dashboard'],
        ['url' => route('role.index'), 'label' => 'Role'],
        ['url' => route('role.show', ['role' => $model->id]), 'label' => $model->name],
    ];

    // Get all permissions for this role with IDs
    $permissionsList = $model->permissions->pluck('name')->toArray();
    $permissionsWithIds = $model->permissions->pluck('id', 'name')->toArray();

    // Get resource labels from config
    $resourceLabels = config('permissions.resources', []);
    $customPermissionsConfig = config('permissions.custom_permissions', []);
    $exclusions = config('permissions.excluded_permissions', []);

    // Initialize all resources with false for all permissions
    $groupedPermissions = [];
    $customPermissionsGrouped = [];

    // Helper function to check if permission is excluded
    $isExcluded = function($resource, $action) use ($exclusions) {
        foreach ($exclusions as $pattern) {
            if ($pattern === "{$resource}.{$action}") return true;
            if (str_starts_with($pattern, '*.') && str_ends_with($pattern, ".{$action}")) return true;
            if ($pattern === "{$resource}.*") return true;
        }
        return false;
    };

    // Initialize all resources from config with false values
    foreach ($resourceLabels as $resourceKey => $resourceLabel) {
        // Skip if entire resource is excluded
        if ($isExcluded($resourceKey, '*')) {
            continue;
        }

        // Initialize basic permissions
        $groupedPermissions[$resourceLabel] = [
            'index' => false,
            'show' => false,
            'create' => false,
            'update' => false,
            'destroy' => false,
        ];

        // Initialize custom permissions for this resource if they exist
        if (isset($customPermissionsConfig[$resourceKey])) {
            $customPermissionsGrouped[$resourceLabel] = [];
            foreach ($customPermissionsConfig[$resourceKey] as $action => $label) {
                if (!$isExcluded($resourceKey, $action)) {
                    $customPermissionsGrouped[$resourceLabel][] = [
                        'resource_key' => $resourceKey,
                        'action' => $action,
                        'label' => $label,
                        'granted' => false
                    ];
                }
            }
        }
    }

    // Now mark the permissions that are actually granted
    foreach ($permissionsList as $permission) {
        $parts = explode('.', $permission);
        if (count($parts) >= 2) {
            $resourceKey = $parts[0];
            $action = $parts[1];

            // Get resource label from config
            $resourceLabel = $resourceLabels[$resourceKey] ?? null;
            if (!$resourceLabel) continue;

            // Check if this is a custom permission
            $isCustom = isset($customPermissionsConfig[$resourceKey]) &&
                       array_key_exists($action, $customPermissionsConfig[$resourceKey]);

            if ($isCustom) {
                // Mark custom permission as granted
                if (isset($customPermissionsGrouped[$resourceLabel])) {
                    foreach ($customPermissionsGrouped[$resourceLabel] as $key => &$customPerm) {
                        if ($customPerm['action'] === $action) {
                            $customPerm['granted'] = true;
                            $customPerm['permission_id'] = $permissionsWithIds[$permission] ?? null;
                            break;
                        }
                    }
                    unset($customPerm); // Break the reference to avoid issues
                }
            } else {
                // Mark standard CRUD permission as granted
                if (isset($groupedPermissions[$resourceLabel])) {
                    if ($action === 'index') {
                        $groupedPermissions[$resourceLabel]['index'] = true;
                    } elseif ($action === 'show') {
                        $groupedPermissions[$resourceLabel]['show'] = true;
                    } elseif (in_array($action, ['create', 'store'])) {
                        $groupedPermissions[$resourceLabel]['create'] = true;
                    } elseif (in_array($action, ['edit', 'update'])) {
                        $groupedPermissions[$resourceLabel]['update'] = true;
                    } elseif ($action === 'destroy') {
                        $groupedPermissions[$resourceLabel]['destroy'] = true;
                    }
                }
            }
        }
    }

    // Sort by resource label
    ksort($groupedPermissions);
    ksort($customPermissionsGrouped);
@endphp

@section('content')
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Role Details</h5>
            @can('role.update')
                <a href="{{ route('role.edit', ['role' => $model->id]) }}" class="btn btn-sm btn-primary">
                    Edit
                </a>
            @endcan
        </div>
        <div class="card-body">
            <div class="row">
                <!-- Role Information -->
                <div class="col-12">
                    <div class="table-responsive">
                        <table class="table table-sm table-borderless small">
                            <tbody>
                                <tr>
                                    <td width="200" class="text-muted">Role Name</td>
                                    <td><strong>{{ $model->name }}</strong></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Audit Information -->
                <div class="col-12 mt-4 pt-3 border-top">
                    <div class="row">
                        <div class="col-md-6">
                            <small class="text-muted">
                                <strong>Created:</strong> {{ $model->created_at?->format('d/m/Y H:i') ?? '-' }}
                                @if ($model->createdBy)
                                    by {{ $model->createdBy->name }}
                                @endif
                            </small>
                        </div>
                        <div class="col-md-6 text-md-end">
                            <small class="text-muted">
                                <strong>Last Updated:</strong> {{ $model->updated_at?->format('d/m/Y H:i') ?? '-' }}
                                @if ($model->updatedBy)
                                    by {{ $model->updatedBy->name }}
                                @endif
                            </small>
                        </div>
                    </div>
                </div>

                <!-- Basic Permissions Matrix -->
                <div class="col-12 mt-4">
                    <h6 class="fw-semibold text-primary mb-3">Basic Permissions</h6>

                    @if (count($groupedPermissions) > 0)
                        <div class="table-responsive">
                            <table class="table table-sm table-hover table-bordered align-middle">
                                <thead class="table-light">
                                    <tr>
                                        <th style="width: 25%;" class="fw-semibold">Module</th>
                                        <th style="width: 15%;" class="text-center fw-semibold">View All</th>
                                        <th style="width: 15%;" class="text-center fw-semibold">View Detail</th>
                                        <th style="width: 15%;" class="text-center fw-semibold">Create</th>
                                        <th style="width: 15%;" class="text-center fw-semibold">Update</th>
                                        <th style="width: 15%;" class="text-center fw-semibold">Delete</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach ($groupedPermissions as $resourceLabel => $permissions)
                                        <tr>
                                            <td class="fw-medium">{{ $resourceLabel }}</td>
                                            <td class="text-center">
                                                @if ($permissions['index'])
                                                    <span class="text-success fw-bold">&#10004;</span>
                                                @else
                                                    <span class="text-danger">&#10006;</span>
                                                @endif
                                            </td>
                                            <td class="text-center">
                                                @if ($permissions['show'])
                                                    <span class="text-success fw-bold">&#10004;</span>
                                                @else
                                                    <span class="text-danger">&#10006;</span>
                                                @endif
                                            </td>
                                            <td class="text-center">
                                                @if ($permissions['create'])
                                                    <span class="text-success fw-bold">&#10004;</span>
                                                @else
                                                    <span class="text-danger">&#10006;</span>
                                                @endif
                                            </td>
                                            <td class="text-center">
                                                @if ($permissions['update'])
                                                    <span class="text-success fw-bold">&#10004;</span>
                                                @else
                                                    <span class="text-danger">&#10006;</span>
                                                @endif
                                            </td>
                                            <td class="text-center">
                                                @if ($permissions['destroy'])
                                                    <span class="text-success fw-bold">&#10004;</span>
                                                @else
                                                    <span class="text-danger">&#10006;</span>
                                                @endif
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @endif
                </div>

                <!-- Custom Permissions Matrix -->
                @php
                    // Filter to only show modules that have at least one granted custom permission
                    $grantedCustomPermissions = collect($customPermissionsGrouped)
                        ->map(function($customPerms) {
                            $filtered = collect($customPerms)->filter(fn($perm) => $perm['granted']);
                            // Remove duplicates by permission action
                            $unique = $filtered->unique('action')->values()->all();
                            return $unique;
                        })
                        ->filter(fn($perms) => count($perms) > 0);
                @endphp
                @if ($grantedCustomPermissions->count() > 0)
                    <div class="col-12 mt-4">
                        <h6 class="fw-semibold text-primary mb-3">Custom Permissions</h6>

                        <div class="table-responsive">
                            <table class="table table-sm table-hover table-bordered align-middle">
                                <thead class="table-light">
                                    <tr>
                                        <th style="width: 25%;" class="fw-semibold">Module</th>
                                        <th class="fw-semibold">Custom Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach ($grantedCustomPermissions as $resourceLabel => $customPerms)
                                        <tr>
                                            <td class="fw-medium">{{ $resourceLabel }}</td>
                                            <td>
                                                @foreach ($customPerms as $customPerm)
                                                    <span class="badge bg-label-success me-2 mb-1" data-permission="{{ $customPerm['resource_key'] }}.{{ $customPerm['action'] }}" data-permission-id="{{ $customPerm['permission_id'] ?? 'N/A' }}" title="ID: {{ $customPerm['permission_id'] ?? 'N/A' }} | Permission: {{ $customPerm['resource_key'] }}.{{ $customPerm['action'] }}">
                                                        <span class="fw-bold">&#10004;</span> {{ $customPerm['label'] }}
                                                    </span>
                                                @endforeach
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    </div>
                @endif
            </div>
        </div>
    </div>
@endsection

<style>
    .table-hover tbody tr:hover {
        background-color: rgba(67, 89, 113, 0.02);
    }
</style>
