@extends('layouts.layoutMaster')

@section('title', ($model->id ? 'Update' : 'Create') . ' Sales Order')

@section('vendor-style')
    @vite('resources/assets/vendor/libs/select2/select2.scss')
@endsection

@section('vendor-script')
    @vite('resources/assets/vendor/libs/select2/select2.js')
@endsection


@php
$m = 'App\\Models\\SalesOrder';
$labels = Helper::getModelLabels($m);

$breadcrumbs = [
        ['url' => route('dashboard'), 'label' => 'Dashboard'],
        ['url' => route('sales-order.index'), 'label' => 'Sales Order'],
    ];

    if ($model->id) {
        $breadcrumbs[] = [
            'url' => route('sales-order.show', ['sales_order' => $model->id]),
            'label' => $model->code,
        ];
        $breadcrumbs[] = [
            'url' => route('sales-order.edit', ['sales_order' => $model->id]),
            'label' => 'Update Sales Order',
        ];
    } else {
        $breadcrumbs[] = [
            'url' => route('sales-order.create'),
            'label' => 'Create Sales Order',
        ];
    }
@endphp

@section('content')
    <div class="row form-row-with-sticky-actions">
        <div class="col-12">
            @if(!$model->id && isset($quotation))
                <div class="alert alert-info d-flex align-items-center mb-3">
                    <i class="icon-base ti tabler-info-circle me-2"></i>
                    <div>
                        <strong>Creating Sales Order from Quotation:</strong> {{ $quotation->code }}
                    </div>
                </div>
            @endif

            <div class="card shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h5 class="mb-1">{{ $model->id ? 'Edit' : 'New' }} Sales Order</h5>
                            <p class="text-muted mb-0 small">
                                {{ $model->id ? 'Update sales order information' : 'Fill in the details below to create a new sales order' }}
                            </p>
                        </div>
                    </div>
                </div>
                <div class="card-body pt-4">
                    <form id="sales-order-form" method="POST"
                        action="{{ $model->id ? route('sales-order.update', $model->id) : route('sales-order.store') }}">
                        @csrf
                        @if ($model->id)
                            @method('PUT')
                        @else
                            <input type="hidden" name="quotation_id" value="{{ $quotation->id ?? '' }}">
                        @endif

                        <div class="row g-3">
                            @if ($isNotSales)
                                <!-- Sales Selection (Only for non-SALES users) -->
                                <div class="col-md-4">
                                    <label for="id_sales" class="form-label">
                                        {{ $labels['id_sales'] ?? 'Sales' }} <span class="text-danger">*</span>
                                    </label>
                                    <select class="form-select select2 @error('id_sales') is-invalid @enderror" id="id_sales"
                                        name="id_sales" required>
                                        <option value="">Select Sales</option>
                                        @foreach ($salesUsers as $salesUser)
                                            <option value="{{ $salesUser->id }}"
                                                {{ old('id_sales', $model->id ? $model->id_sales : ($quotation->id_sales ?? '')) == $salesUser->id ? 'selected' : '' }}>
                                                {{ $salesUser->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('id_sales')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            @endif

                            <!-- Customer Selection -->
                            <div class="col-md-{{ $isNotSales ? '4' : '6' }}">
                                <label for="id_customer" class="form-label">
                                    {{ $labels['id_customer'] ?? 'Customer' }} <span class="text-danger">*</span>
                                </label>
                                <select class="form-select select2 @error('id_customer') is-invalid @enderror" id="id_customer"
                                    name="id_customer" required data-customers='@json($customers->keyBy('id')->map(fn($c) => ['is_ppn' => $c->is_ppn, 'is_pph' => $c->is_pph]))'>
                                    <option value="">Select Customer</option>
                                    @foreach ($customers as $customer)
                                        <option value="{{ $customer->id }}" data-is-ppn="{{ $customer->is_ppn }}"
                                            data-is-pph="{{ $customer->is_pph }}"
                                            {{ old('id_customer', $model->id ? $model->id_customer : ($quotation->id_customer ?? '')) == $customer->id ? 'selected' : '' }}>
                                            {{ $customer->full_name }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('id_customer')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Invoice Date -->
                            <div class="col-md-{{ $isNotSales ? '4' : '6' }}">
                                <label for="invoice_date" class="form-label">
                                    {{ $labels['invoice_date'] ?? 'Invoice Date' }} <span class="text-danger">*</span>
                                </label>
                                <input type="date" class="form-control @error('invoice_date') is-invalid @enderror"
                                    id="invoice_date" name="invoice_date"
                                    value="{{ old('invoice_date', $model->id ? ($model->invoice_date ? $model->invoice_date->format('Y-m-d') : date('Y-m-d')) : ($quotation->invoice_date ? $quotation->invoice_date->format('Y-m-d') : date('Y-m-d'))) }}"
                                    required>
                                @error('invoice_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Notes -->
                            <div class="col-12">
                                <label for="notes" class="form-label">{{ $labels['notes'] ?? 'Notes' }}</label>
                                <textarea class="form-control @error('notes') is-invalid @enderror" id="notes" name="notes" rows="3"
                                    placeholder="Additional notes for this sales order">{{ old('notes', $model->id ? $model->notes : ($quotation->notes ?? '')) }}</textarea>
                                @error('notes')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Divider -->
                            <div class="col-12">
                                <div class="border-top pt-3 mt-2">
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <h6 class="text-muted mb-0">Transaction Details</h6>
                                        <button type="button" class="btn btn-sm btn-primary" id="add-detail-btn" disabled>
                                            <i class="icon-base ti tabler-plus"></i> Add Detail
                                        </button>
                                    </div>
                                </div>
                            </div>

                            <!-- Details Table -->
                            <div class="col-12">
                                <div class="table-responsive" style="overflow: visible;">
                                    <table class="table table-bordered" id="details-table">
                                        <thead class="table-light">
                                            <tr>
                                                <th style="width: 20%;">Route</th>
                                                <th style="width: 15%;">Loadout</th>
                                                <th style="width: 10%;">Quantity</th>
                                                <th style="width: 15%;">Price</th>
                                                <th style="width: 15%;">Sub Total</th>
                                                <th style="width: 10%;">Origin Action</th>
                                                <th style="width: 10%;">Dest. Action</th>
                                                <th style="width: 5%;"></th>
                                            </tr>
                                        </thead>
                                        <tbody id="details-tbody">
                                            <!-- Details will be added here dynamically -->
                                        </tbody>
                                        <tfoot>
                                            <tr>
                                                <td colspan="4" class="text-end"><strong>Total:</strong></td>
                                                <td><strong id="grand-total">0</strong></td>
                                                <td colspan="3"></td>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>

                            <!-- Tax and Discount Section -->
                            <div class="col-12">
                                <div class="border-top pt-3 mt-2">
                                    <h6 class="text-muted mb-3">Tax & Discount</h6>
                                </div>
                            </div>

                            <div class="col-md-4">
                                <label for="discount" class="form-label">{{ $labels['discount'] ?? 'Discount' }}</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rp</span>
                                    <input type="text" class="form-control @error('discount') is-invalid @enderror"
                                        id="discount" name="discount" value="{{ old('discount', $model->id ? $model->discount : ($quotation->discount ?? 0)) }}"
                                        data-format="number" data-submit-raw="true">
                                    @error('discount')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-4" id="ppn-wrapper" style="display: none;">
                                <label for="ppn" class="form-label">{{ $labels['ppn'] ?? 'PPN' }} ({{ $ppnPercentage }}%)</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rp</span>
                                    <input type="text" class="form-control @error('ppn') is-invalid @enderror"
                                        id="ppn" name="ppn" value="{{ old('ppn', $model->id ? $model->ppn : ($quotation->ppn ?? 0)) }}"
                                        data-format="number" readonly disabled>
                                    @error('ppn')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-4" id="pph-wrapper" style="display: none;">
                                <label for="pph" class="form-label">{{ $labels['pph'] ?? 'PPh' }} ({{ $pphPercentage }}%)</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rp</span>
                                    <input type="text" class="form-control @error('pph') is-invalid @enderror"
                                        id="pph" name="pph" value="{{ old('pph', $model->id ? $model->pph : ($quotation->pph ?? 0)) }}"
                                        data-format="number" readonly disabled>
                                    @error('pph')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-12">
                                <div class="alert alert-info d-flex align-items-center">
                                    <i class="icon-base ti tabler-info-circle me-2"></i>
                                    <div>
                                        <strong>Grand Total: Rp <span id="final-grand-total">0</span></strong>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Form Actions - Sticky to bottom -->
    <div class="form-actions-sticky">
        <a href="{{ $isBackUrl }}" class="btn btn-sm btn-label-secondary">
            Cancel
        </a>
        <button type="submit" form="sales-order-form" class="btn btn-sm btn-primary">
            {{ $model->id ? 'Update' : 'Create' }} Sales Order
        </button>
    </div>

    <!-- Detail Row Template -->
    <template id="detail-row-template">
        <tr class="detail-row">
            <input type="hidden" name="details[__INDEX__][id]" value="" class="detail-id-input">
            <input type="hidden" class="origin-hidden" name="details[__INDEX__][id_origin]" value="">
            <input type="hidden" class="origin-name-hidden" name="details[__INDEX__][origin_name]" value="">
            <input type="hidden" class="destination-hidden" name="details[__INDEX__][id_destination]" value="">
            <input type="hidden" class="destination-name-hidden" name="details[__INDEX__][destination_name]" value="">
            <td>
                <select class="form-select form-select-sm select2 route-select" name="details[__INDEX__][route]" required>
                    <option value="">Select Route</option>
                    @foreach ($routes as $route)
                        <option value="{{ $route->id }}" data-origin="{{ $route->id_origin }}"
                            data-destination="{{ $route->id_destination }}"
                            data-origin-name="{{ $route->origin->name ?? '' }}"
                            data-destination-name="{{ $route->destination->name ?? '' }}">
                            {{ $route->origin->name ?? '' }} - {{ $route->destination->name ?? '' }}
                        </option>
                    @endforeach
                </select>
            </td>
            <td>
                <select class="form-select form-select-sm select2 loadout-select" name="details[__INDEX__][id_loadout]" required>
                    <option value="">Select Loadout</option>
                    @foreach ($loadouts as $loadout)
                        <option value="{{ $loadout->id }}">{{ $loadout->name }}</option>
                    @endforeach
                </select>
            </td>
            <td>
                <input type="text" class="form-control form-control-sm quantity-input"
                    name="details[__INDEX__][quantity]" value="1" required data-format="number"
                    data-submit-raw="true">
            </td>
            <td>
                <div class="input-group input-group-sm">
                    <input type="text" class="form-control form-control-sm price-input"
                        name="details[__INDEX__][price]" value="0" required data-format="number"
                        data-submit-raw="true">
                    <button type="button" class="btn btn-outline-secondary btn-sm change-price-btn"
                        style="display: none;" title="Change Price">
                        <i class="icon-base ti tabler-refresh"></i>
                    </button>
                </div>
            </td>
            <td>
                <input type="text" class="form-control form-control-sm subtotal-input"
                    name="details[__INDEX__][sub_total]" readonly disabled value="0" data-format="number">
            </td>
            <td>
                <select class="form-select form-select-sm" name="details[__INDEX__][origin_action]" required>
                    <option value="load">Load</option>
                    <option value="unload">Unload</option>
                </select>
            </td>
            <td>
                <select class="form-select form-select-sm" name="details[__INDEX__][destination_action]" required>
                    <option value="unload" selected>Unload</option>
                    <option value="load">Load</option>
                </select>
            </td>
            <td class="text-center">
                <button type="button" class="btn btn-sm btn-outline-primary edit-allowance-btn" title="Driver Allowance">
                    <i class="icon-base ti tabler-wallet"></i>
                </button>
                <button type="button" class="btn btn-sm btn-danger remove-detail-btn">
                    <i class="icon-base ti tabler-trash"></i>
                </button>
                <input type="hidden" class="pocket-money-1-input" name="details[__INDEX__][pocket_money_1]" value="0">
                <input type="hidden" class="pocket-money-2-input" name="details[__INDEX__][pocket_money_2]" value="0">
                <input type="hidden" class="pocket-money-3-input" name="details[__INDEX__][pocket_money_3]" value="0">
                <input type="hidden" class="bonus-input" name="details[__INDEX__][bonus]" value="0">
            </td>
        </tr>
    </template>

    <!-- Price Selection Modal -->
    <div class="modal fade" id="priceSelectionModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Select Price</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <small>Multiple prices found for this route. Please select one:</small>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-hover" id="price-options-table">
                            <thead class="table-light">
                                <tr>
                                    <th style="width: 5%;"></th>
                                    <th style="width: 30%;">Loadout</th>
                                    <th style="width: 25%;">Price</th>
                                    <th style="width: 20%;">Special Customer Price</th>
                                    <th style="width: 20%;">Notes</th>
                                </tr>
                            </thead>
                            <tbody id="price-options-tbody">
                                <!-- Price options will be loaded here -->
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-sm btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-sm btn-primary" id="select-price-btn">Select Price</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Driver Allowance Modal -->
    <div class="modal fade" id="driverAllowanceModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Driver Allowance</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-12">
                            <label for="modal-pocket-money-1" class="form-label">
                                Pocket Money 1 <span class="text-danger">*</span>
                            </label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control" id="modal-pocket-money-1" value="0"
                                    required data-format="number" data-submit-raw="true">
                            </div>
                        </div>

                        <div class="col-12">
                            <label for="modal-pocket-money-2" class="form-label">Pocket Money 2</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control" id="modal-pocket-money-2" value="0"
                                    data-format="number" data-submit-raw="true">
                            </div>
                        </div>

                        <div class="col-12">
                            <label for="modal-pocket-money-3" class="form-label">Pocket Money 3</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control" id="modal-pocket-money-3" value="0"
                                    data-format="number" data-submit-raw="true">
                            </div>
                        </div>

                        <div class="col-12">
                            <label for="modal-bonus" class="form-label">Bonus</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control" id="modal-bonus" value="0"
                                    data-format="number" data-submit-raw="true">
                            </div>
                        </div>

                        <div class="col-12">
                            <div class="alert alert-info d-flex align-items-center mb-0">
                                <i class="icon-base ti tabler-info-circle me-2"></i>
                                <div>
                                    <strong>Total: Rp <span id="modal-allowance-total">0</span></strong>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-sm btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-sm btn-primary" id="save-allowance-btn">Save</button>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('page-script')
    <script>
        let detailIndex = 0;
        let currentRow = null;
        let priceSelectionModal = null;
        let driverAllowanceModal = null;
        const pphPercentage = {{ $pphPercentage }};
        const ppnPercentage = {{ $ppnPercentage }};
        let customerIsPph = false;
        let customerIsPpn = false;

        // Wait for DOM and jQuery to be ready
        document.addEventListener('DOMContentLoaded', function() {
            // Wait for jQuery to be available
            function initializeWhenReady() {
                if (typeof jQuery === 'undefined') {
                    setTimeout(initializeWhenReady, 50);
                    return;
                }

                // jQuery is ready, proceed with initialization
                $(document).ready(function() {
                    priceSelectionModal = new bootstrap.Modal(document.getElementById('priceSelectionModal'));
                    driverAllowanceModal = new bootstrap.Modal(document.getElementById('driverAllowanceModal'));

                    // Initialize ALL Select2 elements on the page
                    $('.select2').each(function() {
                        var $this = $(this);
                        if (!$this.hasClass('select2-hidden-accessible')) {
                            $this.wrap('<div class="position-relative"></div>').select2({
                                placeholder: 'Select value',
                                dropdownParent: $this.parent(),
                                width: '100%'
                            });
                        }
                    });

                    initializeCustomerHandlers();
                });
            }

            initializeWhenReady();
        });

        function initializeCustomerHandlers() {
            // Check if customer is already selected on page load
            const customerSelect = document.getElementById('id_customer');
            if (customerSelect.value) {
                updateCustomerTaxSettings(customerSelect);
                document.getElementById('add-detail-btn').disabled = false;
            }

            // Customer change (using Select2 event for better compatibility)
            $('#id_customer').on('select2:select', function(e) {
                const previousValue = $(this).data('previousValue') || '';
                const hasDetails = document.getElementById('details-tbody').children.length > 0;
                const currentValue = $(this).val();

                // Only show confirmation if customer was previously selected and there are details filled
                if (previousValue && hasDetails) {
                    if (confirm('Changing customer will clear all details. Continue?')) {
                        document.getElementById('details-tbody').innerHTML = '';
                        updateCustomerTaxSettings(this);
                        document.getElementById('add-detail-btn').disabled = !currentValue;
                        calculateTotal();
                    } else {
                        // Revert to previous selection
                        $(this).val(previousValue).trigger('change.select2');
                        return;
                    }
                } else {
                    updateCustomerTaxSettings(this);
                    document.getElementById('add-detail-btn').disabled = !currentValue;
                }
            });

            // Also handle clearing the select
            $('#id_customer').on('select2:clear', function() {
                document.getElementById('add-detail-btn').disabled = true;
                updateCustomerTaxSettings(this);
            });

            // Store previous value (using jQuery for Select2 compatibility)
            $('#id_customer').on('select2:opening', function() {
                $(this).data('previousValue', $(this).val());
            });

            // Route selection change (delegated with jQuery for Select2 compatibility)
            $('#details-tbody').on('change', '.route-select', function() {
                handleRouteChange(this);
            });

            // Load existing details (for edit mode or validation errors)
            loadExistingDetails();

            // Add detail button
            document.getElementById('add-detail-btn').addEventListener('click', function() {
                const customerId = document.getElementById('id_customer').value;
                if (!customerId) {
                    alert('Please select a customer first');
                    return;
                }
                addDetailRow();
            });

            // Remove detail button (delegated)
            document.getElementById('details-tbody').addEventListener('click', function(e) {
                if (e.target.closest('.remove-detail-btn')) {
                    e.target.closest('tr').remove();
                    calculateTotal();
                }

                // Change price button (delegated)
                if (e.target.closest('.change-price-btn')) {
                    const row = e.target.closest('tr');
                    const storedPrices = row.dataset.availablePrices;
                    if (storedPrices) {
                        currentRow = row;
                        showPriceSelectionModal(JSON.parse(storedPrices));
                    }
                }

                // Edit allowance button (delegated)
                if (e.target.closest('.edit-allowance-btn')) {
                    const row = e.target.closest('tr');
                    currentRow = row;
                    showDriverAllowanceModal(row);
                }
            });

            // Quantity and Price change (delegated)
            document.getElementById('details-tbody').addEventListener('input', function(e) {
                if (e.target.classList.contains('quantity-input') || e.target.classList.contains(
                        'price-input')) {
                    const row = e.target.closest('tr');
                    calculateRowSubtotal(row);
                }
            });

            // Discount change
            document.getElementById('discount').addEventListener('input', function() {
                calculateTotal();
            });

            // Driver allowance modal input changes
            ['modal-pocket-money-1', 'modal-pocket-money-2', 'modal-pocket-money-3', 'modal-bonus'].forEach(id => {
                document.getElementById(id).addEventListener('input', function() {
                    calculateModalAllowanceTotal();
                });
            });

            // Save allowance button
            document.getElementById('save-allowance-btn').addEventListener('click', function() {
                saveDriverAllowance();
            });
        }

        function updateCustomerTaxSettings(selectElement) {
            const selectedOption = selectElement.options[selectElement.selectedIndex];
            if (selectedOption.value) {
                customerIsPph = selectedOption.dataset.isPph === '1';
                customerIsPpn = selectedOption.dataset.isPpn === '1';

                // Show/hide tax fields based on customer settings
                document.getElementById('pph-wrapper').style.display = customerIsPph ? 'block' : 'none';
                document.getElementById('ppn-wrapper').style.display = customerIsPpn ? 'block' : 'none';

                // Reset tax values
                if (!customerIsPph) {
                    document.getElementById('pph').value = 0;
                }
                if (!customerIsPpn) {
                    document.getElementById('ppn').value = 0;
                }

                calculateTotal();
            }
        }

        function addDetailRow() {
            const template = document.getElementById('detail-row-template');
            const clone = template.content.cloneNode(true);
            const html = clone.querySelector('tr').outerHTML.replace(/__INDEX__/g, detailIndex);

            document.getElementById('details-tbody').insertAdjacentHTML('beforeend', html);

            // Initialize number formatting for the new row
            const newRow = document.getElementById('details-tbody').lastElementChild;
            if (window.initNumberFormatting) {
                window.initNumberFormatting(newRow);
            }

            // Initialize Select2 for the new row's select elements
            $(newRow).find('.select2').each(function() {
                var $this = $(this);
                $this.wrap('<div class="position-relative"></div>').select2({
                    placeholder: 'Select value',
                    dropdownParent: $this.parent(),
                    width: '100%'
                });
            });

            detailIndex++;
        }

        function handleRouteChange(selectElement) {
            const row = selectElement.closest('tr');
            const selectedOption = selectElement.options[selectElement.selectedIndex];
            const originId = selectedOption.dataset.origin;
            const destinationId = selectedOption.dataset.destination;
            const originName = selectedOption.dataset.originName;
            const destinationName = selectedOption.dataset.destinationName;

            // Set origin and destination in hidden inputs
            const originHidden = row.querySelector('.origin-hidden');
            const destinationHidden = row.querySelector('.destination-hidden');
            const originNameHidden = row.querySelector('.origin-name-hidden');
            const destinationNameHidden = row.querySelector('.destination-name-hidden');

            if (originId) {
                originHidden.value = originId;
            }
            if (destinationId) {
                destinationHidden.value = destinationId;
            }
            if (originName) {
                originNameHidden.value = originName;
            }
            if (destinationName) {
                destinationNameHidden.value = destinationName;
            }

            // Fetch prices
            const customerId = document.getElementById('id_customer').value;
            const routeId = selectElement.value;

            if (!customerId) {
                alert('Please select a customer first');
                selectElement.value = '';
                return;
            }

            if (routeId) {
                fetchPricesForRoute(routeId, customerId, row);
            }
        }

        function fetchPricesForRoute(routeId, customerId, row) {
            fetch(`/transaction/sales-order/get-prices?route_id=${routeId}&customer_id=${customerId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.data.length > 0) {
                        // Store prices data in row for later use
                        row.dataset.availablePrices = JSON.stringify(data.data);

                        // Show/hide change price button based on number of prices
                        const changePriceBtn = row.querySelector('.change-price-btn');
                        if (data.data.length >= 2) {
                            changePriceBtn.style.display = 'block';
                        } else {
                            changePriceBtn.style.display = 'none';
                        }

                        if (data.data.length === 1) {
                            // Only one price, use it directly
                            applyPriceToRow(row, data.data[0]);
                        } else {
                            // Multiple prices, show modal
                            currentRow = row;
                            showPriceSelectionModal(data.data);
                        }
                    } else {
                        // No prices found, allow manual entry
                        row.querySelector('.price-input').value = 0;
                        row.querySelector('.price-input').focus();
                        // Hide change price button
                        row.querySelector('.change-price-btn').style.display = 'none';
                    }
                })
                .catch(error => {
                    console.error('Error fetching prices:', error);
                    alert('Error fetching prices. Please enter manually.');
                });
        }

        function showPriceSelectionModal(prices) {
            const tbody = document.getElementById('price-options-tbody');
            tbody.innerHTML = '';

            const selectedCustomerId = document.getElementById('id_customer').value;

            prices.forEach((price, index) => {
                const isSpecialCustomerPrice = price.id_customer && price.id_customer == selectedCustomerId;
                const formattedPrice = window.formatNumber ? window.formatNumber(price.price, 0) :
                    formatNumberLocal(price.price);

                const row = `
                    <tr style="cursor: pointer;" onclick="selectPriceRow(${index})">
                        <td class="text-center">
                            <input class="form-check-input" type="radio" name="selectedPrice"
                                value="${index}" ${index === 0 ? 'checked' : ''}>
                        </td>
                        <td>${price.loadout ? price.loadout.name : 'N/A'}</td>
                        <td>Rp ${formattedPrice}</td>
                        <td class="text-center">
                            ${isSpecialCustomerPrice ? '<span class="badge bg-success">Yes</span>' : '<span class="badge bg-secondary">No</span>'}
                        </td>
                        <td>${price.notes ? price.notes : '-'}</td>
                    </tr>
                `;
                tbody.insertAdjacentHTML('beforeend', row);
            });

            // Store prices array for later use
            window.currentPrices = prices;

            // Setup select button handler
            document.getElementById('select-price-btn').onclick = function() {
                const selectedIndex = document.querySelector('input[name="selectedPrice"]:checked')?.value;
                if (selectedIndex !== undefined) {
                    applyPriceToRow(currentRow, prices[selectedIndex]);
                    priceSelectionModal.hide();
                }
            };

            priceSelectionModal.show();
        }

        function selectPriceRow(index) {
            const radio = document.querySelector(`input[name="selectedPrice"][value="${index}"]`);
            if (radio) {
                radio.checked = true;
            }
        }

        function applyPriceToRow(row, priceData) {
            // Set loadout using jQuery for Select2
            const loadoutSelect = $(row).find('.loadout-select');
            loadoutSelect.val(priceData.id_loadout).trigger('change.select2');

            const priceInput = row.querySelector('.price-input');

            // Set raw price value without triggering format
            // Remove any existing formatting before setting
            const rawPrice = priceData.price;
            priceInput.value = rawPrice;

            // Format the price manually using the global formatter if available
            if (window.formatNumber) {
                priceInput.value = window.formatNumber(rawPrice, 0);
            }

            calculateRowSubtotal(row);
        }

        function calculateRowSubtotal(row) {
            const quantityInput = row.querySelector('.quantity-input');
            const priceInput = row.querySelector('.price-input');
            const subtotalInput = row.querySelector('.subtotal-input');

            // Parse formatted numbers
            const quantity = window.parseNumber ? window.parseNumber(quantityInput.value) : parseFloat(quantityInput
                .value) || 0;
            const price = window.parseNumber ? window.parseNumber(priceInput.value) : parseFloat(priceInput.value) || 0;
            const subtotal = quantity * price;

            // Format subtotal
            subtotalInput.value = window.formatNumber ? window.formatNumber(subtotal, 0) : subtotal.toFixed(2);
            calculateTotal();
        }

        function calculateTotal() {
            let total = 0;
            document.querySelectorAll('.subtotal-input').forEach(input => {
                const value = window.parseNumber ? window.parseNumber(input.value) : parseFloat(input.value) || 0;
                total += value;
            });

            const customerId = document.getElementById('id_customer').value;
            let ppn = 0;
            let pph = 0;

            const discountInput = document.getElementById('discount');
            const discount = window.parseNumber ? window.parseNumber(discountInput.value) : parseFloat(discountInput
                .value) || 0;
            const afterDiscount = total - discount;

            // Calculate PPN and PPh based on customer settings and config percentages
            if (customerIsPpn) {
                ppn = afterDiscount * (ppnPercentage / 100);
            }
            if (customerIsPph) {
                pph = afterDiscount * (pphPercentage / 100);
            }

            const ppnInput = document.getElementById('ppn');
            const pphInput = document.getElementById('pph');

            // Use global formatter if available with 2 decimal places
            if (window.formatNumber) {
                ppnInput.value = window.formatNumber(ppn, 2);
                pphInput.value = window.formatNumber(pph, 2);
            } else {
                ppnInput.value = ppn.toFixed(2);
                pphInput.value = pph.toFixed(2);
            }

            const grandTotal = afterDiscount + ppn - pph;

            // Format display
            const formattedTotal = window.formatNumber ? window.formatNumber(total, 0) : formatNumberLocal(total.toFixed(
                2));
            const formattedGrandTotal = window.formatNumber ? window.formatNumber(grandTotal, 0) : formatNumberLocal(
                grandTotal.toFixed(2));

            document.getElementById('grand-total').textContent = formattedTotal;
            document.getElementById('final-grand-total').textContent = formattedGrandTotal;
        }

        function formatNumberLocal(num) {
            return parseFloat(num).toLocaleString('id-ID', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        }

        function loadExistingDetails() {
            // Prepare details data from either old input (validation error) or model (edit mode)
            @if($model->id)
                const existingDetails = @json(old('details', $model->details ?? []));
            @else
                const existingDetails = @json(old('details', $quotation->details ?? []));
            @endif

            if (!existingDetails || existingDetails.length === 0) {
                return;
            }

            // Load each detail row
            existingDetails.forEach((detail, index) => {
                addDetailRow();
                const row = document.getElementById('details-tbody').lastElementChild;

                // Set detail ID if it exists (for updates)
                if (detail.id) {
                    const detailIdInput = row.querySelector('.detail-id-input');
                    if (detailIdInput) {
                        detailIdInput.value = detail.id;
                    }
                }

                // Find route by origin and destination
                let routeId = null;
                const routeSelect = row.querySelector('.route-select');

                // Loop through route options to find matching origin and destination
                for (let i = 0; i < routeSelect.options.length; i++) {
                    const option = routeSelect.options[i];
                    if (option.dataset.origin == detail.id_origin &&
                        option.dataset.destination == detail.id_destination) {
                        routeId = option.value;
                        break;
                    }
                }

                // Set origin and destination in hidden inputs first
                const originHidden = row.querySelector('.origin-hidden');
                const destinationHidden = row.querySelector('.destination-hidden');
                const originNameHidden = row.querySelector('.origin-name-hidden');
                const destinationNameHidden = row.querySelector('.destination-name-hidden');

                originHidden.value = detail.id_origin;
                destinationHidden.value = detail.id_destination;
                originNameHidden.value = detail.origin_name || '';
                destinationNameHidden.value = detail.destination_name || '';

                // Set route - Use jQuery to set value and trigger change for Select2
                if (routeId) {
                    // Set the value without triggering the change event first (to avoid fetching prices)
                    $(routeSelect).val(routeId).trigger('change.select2');
                }

                // Set loadout - Use jQuery to set value and trigger change for Select2
                const loadoutSelect = row.querySelector('.loadout-select');
                $(loadoutSelect).val(detail.id_loadout).trigger('change.select2');

                // Set quantity and format it
                const quantityInput = row.querySelector('.quantity-input');
                quantityInput.value = window.formatter ? window.formatter(detail.quantity) : detail.quantity;

                // Set price and format it
                const priceInput = row.querySelector('.price-input');
                priceInput.value = window.formatter ? window.formatter(detail.price) : detail.price;

                // Set origin and destination actions
                const originActionSelect = row.querySelector('select[name*="[origin_action]"]');
                const destActionSelect = row.querySelector('select[name*="[destination_action]"]');
                if (originActionSelect) originActionSelect.value = detail.origin_action || 'load';
                if (destActionSelect) destActionSelect.value = detail.destination_action || 'unload';

                // Set driver allowances and format them
                const pocketMoney1Input = row.querySelector('.pocket-money-1-input');
                const pocketMoney2Input = row.querySelector('.pocket-money-2-input');
                const pocketMoney3Input = row.querySelector('.pocket-money-3-input');
                const bonusInput = row.querySelector('.bonus-input');

                pocketMoney1Input.value = window.formatter ? window.formatter(detail.pocket_money_1 || 0) : (detail.pocket_money_1 || 0);
                pocketMoney2Input.value = window.formatter ? window.formatter(detail.pocket_money_2 || 0) : (detail.pocket_money_2 || 0);
                pocketMoney3Input.value = window.formatter ? window.formatter(detail.pocket_money_3 || 0) : (detail.pocket_money_3 || 0);
                bonusInput.value = window.formatter ? window.formatter(detail.bonus || 0) : (detail.bonus || 0);

                // Calculate subtotal for this row
                calculateRowSubtotal(row);
            });

            // Reinitialize number formatting for discount field if needed
            const discountInput = document.getElementById('discount');
            if (window.initNumberFormatting && discountInput) {
                window.initNumberFormatting();
            }

            // Calculate total after all rows are loaded
            calculateTotal();
        }

        function showDriverAllowanceModal(row) {
            // Get current values from row or from route defaults
            const routeSelect = row.querySelector('.route-select');
            const routeId = routeSelect.value;

            // Get current values from hidden inputs
            const pocketMoney1 = window.parseNumber ? window.parseNumber(row.querySelector('.pocket-money-1-input').value) : parseFloat(row.querySelector('.pocket-money-1-input').value) || 0;
            const pocketMoney2 = window.parseNumber ? window.parseNumber(row.querySelector('.pocket-money-2-input').value) : parseFloat(row.querySelector('.pocket-money-2-input').value) || 0;
            const pocketMoney3 = window.parseNumber ? window.parseNumber(row.querySelector('.pocket-money-3-input').value) : parseFloat(row.querySelector('.pocket-money-3-input').value) || 0;
            const bonus = window.parseNumber ? window.parseNumber(row.querySelector('.bonus-input').value) : parseFloat(row.querySelector('.bonus-input').value) || 0;

            // If route is selected and values are all 0, fetch defaults from route
            if (routeId && pocketMoney1 === 0 && pocketMoney2 === 0 && pocketMoney3 === 0 && bonus === 0) {
                fetch(`/transaction/sales-order/get-route-allowances?route_id=${routeId}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            setModalAllowanceValues(data.data);
                        } else {
                            setModalAllowanceValues({
                                pocket_money_1: 0,
                                pocket_money_2: 0,
                                pocket_money_3: 0,
                                bonus: 0
                            });
                        }
                        driverAllowanceModal.show();
                    })
                    .catch(error => {
                        console.error('Error fetching route allowances:', error);
                        setModalAllowanceValues({
                            pocket_money_1: 0,
                            pocket_money_2: 0,
                            pocket_money_3: 0,
                            bonus: 0
                        });
                        driverAllowanceModal.show();
                    });
            } else {
                // Set current values
                setModalAllowanceValues({
                    pocket_money_1: pocketMoney1,
                    pocket_money_2: pocketMoney2,
                    pocket_money_3: pocketMoney3,
                    bonus: bonus
                });
                driverAllowanceModal.show();
            }
        }

        function setModalAllowanceValues(data) {
            const modalPocketMoney1 = document.getElementById('modal-pocket-money-1');
            const modalPocketMoney2 = document.getElementById('modal-pocket-money-2');
            const modalPocketMoney3 = document.getElementById('modal-pocket-money-3');
            const modalBonus = document.getElementById('modal-bonus');

            // Set formatted values
            if (window.formatNumber) {
                modalPocketMoney1.value = window.formatNumber(data.pocket_money_1 || 0);
                modalPocketMoney2.value = window.formatNumber(data.pocket_money_2 || 0);
                modalPocketMoney3.value = window.formatNumber(data.pocket_money_3 || 0);
                modalBonus.value = window.formatNumber(data.bonus || 0);
            } else {
                modalPocketMoney1.value = data.pocket_money_1 || 0;
                modalPocketMoney2.value = data.pocket_money_2 || 0;
                modalPocketMoney3.value = data.pocket_money_3 || 0;
                modalBonus.value = data.bonus || 0;
            }

            calculateModalAllowanceTotal();
        }

        function calculateModalAllowanceTotal() {
            const modalPocketMoney1 = document.getElementById('modal-pocket-money-1');
            const modalPocketMoney2 = document.getElementById('modal-pocket-money-2');
            const modalPocketMoney3 = document.getElementById('modal-pocket-money-3');
            const modalBonus = document.getElementById('modal-bonus');

            const pocketMoney1 = window.parseNumber ? window.parseNumber(modalPocketMoney1.value) : parseFloat(modalPocketMoney1.value) || 0;
            const pocketMoney2 = window.parseNumber ? window.parseNumber(modalPocketMoney2.value) : parseFloat(modalPocketMoney2.value) || 0;
            const pocketMoney3 = window.parseNumber ? window.parseNumber(modalPocketMoney3.value) : parseFloat(modalPocketMoney3.value) || 0;
            const bonus = window.parseNumber ? window.parseNumber(modalBonus.value) : parseFloat(modalBonus.value) || 0;

            const total = pocketMoney1 + pocketMoney2 + pocketMoney3 + bonus;

            const formattedTotal = window.formatNumber ? window.formatNumber(total, 0) : formatNumberLocal(total);
            document.getElementById('modal-allowance-total').textContent = formattedTotal;
        }

        function saveDriverAllowance() {
            if (!currentRow) return;

            const modalPocketMoney1 = document.getElementById('modal-pocket-money-1');
            const modalPocketMoney2 = document.getElementById('modal-pocket-money-2');
            const modalPocketMoney3 = document.getElementById('modal-pocket-money-3');
            const modalBonus = document.getElementById('modal-bonus');

            const pocketMoney1 = window.parseNumber ? window.parseNumber(modalPocketMoney1.value) : parseFloat(modalPocketMoney1.value) || 0;
            const pocketMoney2 = window.parseNumber ? window.parseNumber(modalPocketMoney2.value) : parseFloat(modalPocketMoney2.value) || 0;
            const pocketMoney3 = window.parseNumber ? window.parseNumber(modalPocketMoney3.value) : parseFloat(modalPocketMoney3.value) || 0;
            const bonus = window.parseNumber ? window.parseNumber(modalBonus.value) : parseFloat(modalBonus.value) || 0;

            // Update hidden inputs
            currentRow.querySelector('.pocket-money-1-input').value = pocketMoney1;
            currentRow.querySelector('.pocket-money-2-input').value = pocketMoney2;
            currentRow.querySelector('.pocket-money-3-input').value = pocketMoney3;
            currentRow.querySelector('.bonus-input').value = bonus;

            driverAllowanceModal.hide();
        }
    </script>
@endsection
